/* @(#) $Id: multimaster.cpp 14 2012-01-31 07:15:15Z leres $ (XSE) */

#define __STDC_LIMIT_MACROS
#include <stdint.h>			/* pick up UINT32_MAX */

#if defined(ARDUINO) && ARDUINO >= 100
#include <Arduino.h>
#else
#include <WProgram.h>
#endif

#include <ctype.h>

#include <twi.h>			/* pick up TWI_FREQ */
#include <Wire.h>

#include "version.h"

/* Subtract t2 from t1; handles overflow correctly */
#define MICROS_SUB(t1, t2) \
    (((t1) >= (t2)) ? (t1) - (t2) : (t1) + (UINT32_MAX - (t2)))
#define MILLIS_SUB(t1, t2) MICROS_SUB((t1), (t2))


/* First two digital pins on port A */
/* d0: used for serial0 RX */
/* d1: used for serial0 TX */

/* d11: used for SPI/MOSI */
/* d12: used for SPI/MISO */

#define PIN_LED_OUT	13		/* Also SCK */

/* Last two digital pins on port C */
/* d18/a4: used for TWI/SDA */
/* d19/a5: used for TWI/SCL */

#define LED_SET(v) digitalWrite(PIN_LED_OUT, (v) ? LOW : HIGH)

#ifndef UART0_BAUD
#define UART0_BAUD 57600
#endif

/* Locals */
unsigned long now_msec;			/* ms since boot at top of loop() */

/* Circular input buffer */
char ibuf[512];				/* chars from other master to us */
volatile static char *igp;		/* end pointer: owned by loop()  */
volatile static char *iep;		/* end pointer: owned by onreceive() */

/* Next position in circular input buffer */
#define IBUF_NEXTP(p) (((p) < ibuf + sizeof(ibuf) - 1) ? (p) + 1 : ibuf)

/* Fix up if pointer is past end of buffer */
#define IBUF_WRAP(p) (((p) < ibuf + sizeof(ibuf)) ? (p) : (p) - sizeof(ibuf))

/* Previous position in circular input buffer */
#define IBUF_PREVP(p) ((p != ibuf) ? (p) - 1 : ibuf + sizeof(ibuf) - 1)

/* True of there are characters to be read */
#define IBUF_READY() (igp != iep)

/* Forwards */
static void heartbeat(void);
void loop(void);
void setup(void);
void onreceive(int);

void
loop()
{
	int ch;
	size_t cc;
	char *cp;
	char obuf[BUFFER_LENGTH];	/* chars from us to other master */

	now_msec = millis();

	/* Toggle led */
	heartbeat();

	/* Copy from I2C input buffer to serial */
	while (IBUF_READY()) {
		ch = *igp;
		igp = IBUF_NEXTP(igp);
		Serial.write(ch);
		if (ch == '\r')
			Serial.write('\n');
	}

	/* Copy from serial to local buffer */
	cp = obuf;
	cc = 0;
	while (Serial.available() > 0 && cp < (obuf + sizeof(obuf) - 1)) {
		ch = Serial.read();
		/* Undocumented: -1 when no more characters */
		if (ch < 0)
			break;
		*cp++ = ch;
		++cc;
	}

	/* Write from local buffer to other master (at most BUFFER_LENGTH) */
	if (cc > 0) {
		Wire.beginTransmission(TWI_SLAVE);
		Wire.write((const uint8_t *)obuf, cc);
		Wire.endTransmission();
	}
}

void
setup()
{
	/* Serial initialization */
	Serial.begin(UART0_BAUD);

	/* LED */
	pinMode(PIN_LED_OUT, OUTPUT);

	/* I2C */
	iep = ibuf;
	igp = ibuf;
	*iep = '\0';
	Wire.onReceive(onreceive);
	Wire.begin(TWI_MASTER);

	/* Howdy */
	Serial.print("\r\nI2C Multi-Master Example V");
	Serial.print(version);
	Serial.print(": master");
	Serial.print(TWI_MASTER);
	Serial.print(" @ ");
	Serial.print(TWI_FREQ / 1000);
	Serial.println("kHz");
}

/* on/off times are LEDSTATEMS ms: must be even */
static int8_t ledstate[] = { 1, 2, 1, 36 };
#define LEDSTATESIZE (sizeof(ledstate) / sizeof(ledstate[0]))
#define LEDSTATEMS 25

static void
heartbeat()
{
	static unsigned long last;
	static int16_t left;
	static boolean led;
	static int8_t *p = NULL;

	if (p == NULL) {
		p = ledstate;
		left = 0;
		led = 0;
		last = now_msec;
	}

	left -= MILLIS_SUB(now_msec, last);
	if (left <= 0) {
		LED_SET(led);
		left = *p * LEDSTATEMS;
		++p;
		if (p - ledstate >= LEDSTATESIZE)
			p = ledstate;
		led = !led;
	}
	last = now_msec;
}

/* Receive and store characters from the master */
void
onreceive(int n)
{
	int ch;
	volatile char *p;

	while (n-- > 0) {
		ch = Wire.read();
		p = IBUF_NEXTP(iep);

		/* Add to buffer if there's room */
		if (p != igp) {
			*iep = ch;
			iep = p;
		}
	}
	*iep = '\0';
}
