#ifndef lint
static const char rcsid[] =
    "@(#) $Id: cupdate.c 31 2010-09-02 00:57:20Z leres $ (LBL)";
#endif
/*
 * Copyright (c) 2005, 2006, 2008, 2009, 2010
 *	Craig Leres
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <ctype.h>
#include <curses.h>
#include <errno.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#define CTRL(c) ((c) & 0x1F)

/* Globals */
const char *prog;
int debug;
int verbose;
int waitsecs = 2;
extern const char version[];


/* Forwards */
char *buildcommand(int, char **);
void die(const char *, ...) __attribute__((volatile, format (printf, 1, 2)));
void finish(void);
char *runcommand(char *);
int trimws(char *, int);
void usage(void) __attribute__((volatile));

int
main(int argc, char **argv)
{
	register int op, n;
	register char *cp, *cmd, *out, ch;
	fd_set rfds;
	struct timeval tv;
	WINDOW *win;
	const char *tty;
	struct stat sbuf;

	if (argv[0] == NULL)
		prog = "gearratio";
        else if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	while ((op = getopt(argc, argv, "dvw:")) != EOF)
		switch (op) {

		case 'd':
			++debug;
			break;

		case 'v':
			++verbose;
			break;

		case 'w':
			waitsecs = atoi(optarg);
			if (waitsecs <= 0) {
				fprintf(stderr, "%s: waitsecs must be > 0\n",
				    prog);
				exit(1);
			}
			break;

		default:
			usage();
		}
	if (argc == optind)
		usage();

	/* Build command line */
	cmd = buildcommand(argc - optind, argv + optind);
	if (debug)
		fprintf(stderr, "%s: cmd: %s\n", prog, cmd);

	win = initscr();
	if (win == NULL) {
		fprintf(stderr,
		    "%s: Terminal type unset or lacking necessary features\n",
		    prog);
		exit(1);
	}

	/* Disable messages (if possible) */
	tty = ttyname(STDERR_FILENO);
	if ((tty != NULL && stat(tty, &sbuf) < 0) ||
	    (sbuf.st_mode & S_IWGRP) == 0 ||
	    chmod(tty, sbuf.st_mode & ~S_IWGRP) < 0)
		tty = NULL;

	cbreak();
	noecho();
	clear();
	tv.tv_usec = 0;
	for (;;) {
		out = runcommand(cmd);
		erase();
		addstr(out);
		refresh();
		FD_ZERO(&rfds);
		FD_SET(fileno(stdin), &rfds);
		tv.tv_sec = waitsecs;
		n = select(fileno(stdin) + 1, &rfds, NULL, NULL, &tv);
		if (n < 0) {
			if (errno == EINTR)
				continue;
			die("select: %s", strerror(errno));
		}
		if (n > 0 && FD_ISSET(fileno(stdin), &rfds)) {
			ch = getch();
			if (ch == 'q' || ch == 'Q')
				break;
			if (ch == CTRL('L'))
				clearok(win, 1);
			else if (ch != ERR)
				beep();
		}
	}

	finish();
	free(cmd);

	/* Restore messages */
	if (tty != NULL && chmod(tty, sbuf.st_mode) < 0) {
		fprintf(stderr, "%s: chmod(%s): %s\n",
		    prog, tty, strerror(errno));
		exit(1);
	}

	exit(0);
}

char *
buildcommand(register int ac, register char **av)
{
	register int n, len;
	register char *cp, *cmd;
	static const char redirect[] = " 2>&1";

	/* Room for optional for "(" and ")" */
	len = strlen(redirect) + 2;
	for (n = 0; n < ac; ++n)
		len += strlen(av[n]) + 1;

	cmd = malloc(len);
	if (cmd == 0)
		die("buildcommand: malloc: %s", strerror(errno));

	cp = cmd;
	for (n = 0; n < ac; ++n) {
		strcpy(cp, av[n]);
		cp += strlen(cp);
		*cp++ = ' ';
	}
	if (cp > cmd)
		--cp;

	if (strchr(cmd, ';') != NULL) {
		memmove(cmd + 1, cmd, cp - cmd);
		cmd[0] = '(';
		++cp;
		*cp = ')';
		++cp;
	}
	strcpy(cp, redirect);

	return (cmd);
}

void
finish()
{

	move(LINES - 1, 0);
	clrtobot();
	refresh();
	endwin();
}

char *
runcommand(register char *cmd)
{
	register FILE *p;
	register char *cp;
	register int n, cc, len;
	static int size = 0;
	static char *out = NULL;

	if (size == 0) {
		size = 8;
		out = malloc(size);
		if (out == NULL)
			die("runcommand: malloc: %s", strerror(errno));
	}
	len = 0;

	p = popen(cmd, "r");
	if (p == NULL)
		die("popen failed: %s", strerror(errno));

	for (;;) {
		if (size <= len + 1) {
			size <<= 1;
			out = realloc(out, size);
			if (out == NULL)
				die("runcommand: realloc: %s", strerror(errno));
		}
		cp = out + len;
		n = size - (len + 1);
		cc = fread(cp, 1, n, p);
		if (cc < 0)
			die("runcommand: fread: %s", strerror(errno));
		if (cc == 0) {
			*cp = '\0';
			break;
		}
		cc = trimws(cp, cc);
		cp += cc;
		len += cc;
	}

	pclose(p);
	return (out);
}

void
die(const char *fmt, ...)
{
	va_list ap;

	finish();
	va_start(ap, fmt);
	fprintf(stderr, "%s: ", prog);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	exit(1);
}

int
trimws(char *str, int cc)
{
	char *cp, *cp2, *cp3;
	int i, j;

	cp = str;
	while ((cp2 = strchr(cp, '\n')) != NULL) {
		i = cp2 - cp;
		cp3 = cp2 = cp + i;
		j = 0;
		while (cp2 > cp && isspace((int)cp2[-1])) {
			--cp2;
			++j;
		}
		memmove(cp2, cp2 + j, cp3 - cp2 + 1);
		cp = cp2 + 1;
		cc -= j;
	}

	str[cc] = '\0';
	return (cc);
}

void
usage(void)
{

	fprintf(stderr, "%s version %s\n", prog, version);
	fprintf(stderr, "usage: %s [-dv] [-w waitsecs] command [...]\n", prog);
	exit(1);
}
