#ifndef lint
static const char rcsid[] __attribute__ ((unused)) =
    "@(#) $Id: cupdate.c 53 2023-04-02 22:45:45Z leres $ (LBL)";
#endif
/*
 * Copyright (c) 2005, 2006, 2008, 2009, 2010, 2013, 2016, 2018, 2020, 2021, 2022, 2023
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *
 *  2. Redistributions in binary form must reproduce the above
 *     copyright notice, this list of conditions and the following
 *     disclaimer in the documentation and/or other materials
 *     provided with the distribution.
 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHOR
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS
 * OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>

#include <ctype.h>
#include <curses.h>
#include <errno.h>
#include <signal.h>
#include <stdarg.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "setsignal.h"

#define CTRL(c) ((c) & 0x1F)

/* Globals */
const char *prog;
int debug;
int verbose;
int waitsecs = 1;
int waiting;
int done;
WINDOW *win;
#ifdef SIGWINCH
sig_t oldsigwinch;
#endif
extern const char version[];

/* Forwards */
char *buildcommand(int, char **);
void __attribute__((noreturn, format (printf, 1, 2))) die(const char *, ...);
void finish(void);
void sigint(int);
#ifdef SIGWINCH
void sigwinch(int);
#endif
char *runcommand(char *);
int trimws(char *, int);
void __attribute__((noreturn)) usage(void);

int
main(int argc, char **argv)
{
	int op, n;
	long lv;
	char *cp, *ep, *cmd, *out, ch;
	fd_set rfds;
	struct timeval tv;
	const char *tty;
	struct stat sbuf;

        if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	opterr = 0;
	while ((op = getopt(argc, argv, "dvw:")) != EOF)
		switch (op) {

		case 'd':
			++debug;
			break;

		case 'v':
			++verbose;
			break;

		case 'w':
			lv = strtol(optarg, &ep, 10);
			if (*ep != '\0' || lv <= 0)
				usage();
			waitsecs = lv;
			break;

		default:
			usage();
		}
	if (argc == optind)
		usage();

	/* Build command line */
	cmd = buildcommand(argc - optind, argv + optind);
	if (debug)
		fprintf(stderr, "%s: cmd: %s\n", prog, cmd);

	win = initscr();
	if (win == NULL) {
		fprintf(stderr,
		    "%s: Terminal type unset or lacking necessary features\n",
		    prog);
		exit(1);
	}

	/* Disable messages (if possible) */
	tty = ttyname(STDERR_FILENO);
	if ((tty != NULL && stat(tty, &sbuf) < 0) ||
	    (sbuf.st_mode & S_IWGRP) == 0 ||
	    chmod(tty, sbuf.st_mode & ~S_IWGRP) < 0)
		tty = NULL;

	(void)setsignal(SIGINT, sigint);
#ifdef SIGWINCH
	oldsigwinch = setsignal(SIGWINCH, sigwinch);
#endif

	cbreak();
	noecho();
	clear();
	/* Turn off cursor */
	curs_set(0);
	tv.tv_usec = 0;
	for (;;) {
		out = runcommand(cmd);
		erase();
		addstr(out);
		refresh();
		FD_ZERO(&rfds);
		FD_SET(fileno(stdin), &rfds);
		tv.tv_sec = waitsecs;
		/* Ignore slight race condition */
		++waiting;
		n = select(fileno(stdin) + 1, &rfds, NULL, NULL, &tv);
		waiting = 0;

		/* ^C exit */
		if (done)
			break;

		if (n < 0) {
			if (errno == EINTR)
				continue;
			die("select: %s", strerror(errno));
		}
		if (n > 0 && FD_ISSET(fileno(stdin), &rfds)) {
			ch = getch();
			if (ch == 'q' || ch == 'Q')
				break;
			if (ch == CTRL('L'))
				clearok(win, 1);
			else if (ch != ERR)
				beep();
		}
	}

	finish();
	free(cmd);

	/* Restore messages */
	if (tty != NULL && chmod(tty, sbuf.st_mode) < 0) {
		fprintf(stderr, "%s: chmod(%s): %s\n",
		    prog, tty, strerror(errno));
		exit(1);
	}

	exit(0);
}

char *
buildcommand(int ac, char **av)
{
	int n, len;
	char *cp, *cmd;
	static const char redirect[] = " 2>&1";

	/* Room for optional for "(" and ")" */
	len = strlen(redirect) + 2;
	for (n = 0; n < ac; ++n)
		len += strlen(av[n]) + 1;

	cmd = malloc(len);
	if (cmd == 0)
		die("buildcommand: malloc: %s", strerror(errno));

	cp = cmd;
	for (n = 0; n < ac; ++n) {
		strcpy(cp, av[n]);
		cp += strlen(cp);
		*cp++ = ' ';
	}
	if (cp > cmd)
		--cp;

	if (strchr(cmd, ';') != NULL) {
		memmove(cmd + 1, cmd, cp - cmd);
		cmd[0] = '(';
		++cp;
		*cp = ')';
		++cp;
	}
	strcpy(cp, redirect);

	return (cmd);
}

void
die(const char *fmt, ...)
{
	va_list ap;

	finish();
	va_start(ap, fmt);
	fprintf(stderr, "%s: ", prog);
	vfprintf(stderr, fmt, ap);
	va_end(ap);
	exit(1);
}

void
finish()
{

	move(LINES - 1, 0);
	clrtobot();
	refresh();
	endwin();
}

char *
runcommand(char *cmd)
{
	FILE *p;
	char *cp;
	int n, cc, len;
	static int size = 0;
	static char *out = NULL;

	if (size == 0) {
		size = 8;
		out = malloc(size);
		if (out == NULL)
			die("runcommand: malloc: %s", strerror(errno));
	}
	len = 0;

	p = popen(cmd, "r");
	if (p == NULL)
		die("popen failed: %s", strerror(errno));

	for (;;) {
		if (size <= len + 1) {
			size <<= 1;
			out = realloc(out, size);
			if (out == NULL)
				die("runcommand: realloc: %s", strerror(errno));
		}
		cp = out + len;
		n = size - (len + 1);
		cc = fread(cp, 1, n, p);
		if (cc < 0)
			die("runcommand: fread: %s", strerror(errno));
		if (cc == 0) {
			*cp = '\0';
			break;
		}
		cc = trimws(cp, cc);
		cp += cc;
		len += cc;
	}

	pclose(p);
	return (out);
}

void
sigint(int signo)
{
	++done;
	return;
}

#ifdef SIGWINCH
void
sigwinch(int signo)
{
	(void)oldsigwinch(signo);
	if (waiting && win != NULL) {
		redrawwin(win);
		refresh();
	}
	return;
}
#endif

/* Strip whitespace at the end of each line preserving the \n */
int
trimws(char *str, int cc)
{
	char *cp, *cp2, *cp3;
	int len, j;

	/* Insure termination */
	str[cc] = '\0';
	cp = str;
	len = cc;
	while ((cp2 = strchr(cp, '\n')) != NULL) {
		len -= cp2 - cp;
		cp3 = cp2;
		j = 0;
		while (cp3 > cp && isspace((int)cp3[-1])) {
			--cp3;
			++j;
		}
		if (j > 0) {
			memmove(cp3, cp2, len);
			cc -= j;
		}
		cp = cp2 + 1;
	}

	str[cc] = '\0';
	return (cc);
}

void
usage(void)
{

	fprintf(stderr, "%s version %s\n", prog, version);
	fprintf(stderr, "usage: %s [-dv] [-w waitsecs] command [...]\n", prog);
	exit(1);
}
