/*
 * parseimg - figure out the image size of GIF or JPEG files
 *
 * Copyright (C) 1997
 *	by Jef Poskanzer <jef@acme.com>.  All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
*/

#include <sys/types.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "gnuc.h"
#ifdef HAVE_OS_PROTO_H
#include "os-proto.h"
#endif

#include "parseimg.h"

/* Define a few JPEG markers */
#define M_SOF0 0xc0
#define M_SOF3 0xc3
#define M_SOI 0xd8
#define M_EOI 0xd9
#define M_SOS 0xda

#define CTRL(c) ((c) & 0x1F)

/* Globals */
extern char *prog;

/* Forwards */
static int get(FILE *, u_char *);
static int gif(FILE *, int *, int *);
static int jpeg(FILE *, int *, int *);
static int mustbe(FILE *, u_char);
static int png(FILE *, int *, int *);

int
parseimg(register char *image, register int *wp, register int *hp)
{
	register int status;
	register FILE *f;
	u_char ch;

	if (*image == '\0' || strncasecmp(image, "http://", 7) == 0)
		return (0);

	f = fopen(image, "r");
	if (f == NULL) {
		fprintf(stderr, "%s: fopen %s: %s\n",
		    prog, image, strerror(errno));
		return (0);
	}

	if (!get(f, &ch)) {
		fclose(f);
		return (0);
	}
	if (ch == 'G')
		status = gif(f, wp, hp);
	else if (ch == 0xff)
		status = jpeg(f, wp, hp);
	else if (ch == 0x89)
		status = png(f, wp, hp);
	else
		status = 0;
	fclose(f);
	return (status);
}



static int
gif(register FILE * f, register int *wp, register int *hp)
{
	register int w, h;
	u_char ch, w1, w2, h1, h2;

	if (!mustbe(f, 'I'))
		return (0);
	if (!mustbe(f, 'F'))
		return (0);
	if (!mustbe(f, '8'))
		return (0);
	if (!get(f, &ch))
		return (0);
	if (ch != '7' && ch != '9')
		return (0);
	if (!mustbe(f, 'a'))
		return (0);

	/* Width and height are the next things in the file. */
	if (!get(f, &w1))
		return (0);
	if (!get(f, &w2))
		return (0);
	if (!get(f, &h1))
		return (0);
	if (!get(f, &h2))
		return (0);
	w = (int) w2 * 256 + (int) w1;
	h = (int) h2 * 256 + (int) h1;
	*wp = w;
	*hp = h;
	return (1);
}


static int
jpeg(register FILE *f, register int *wp, register int *hp)
{
	register int l, w, h;
	register int i;
	u_char ch, l1, l2, w1, w2, h1, h2;

	/* Read rest of initial marker. */
	if (!mustbe(f, M_SOI))
		return (0);

	/*
	 * JPEG blocks consist of a 0xff, a marker byte, a block size
	 * (two bytes, big-endian), and the rest of the block. The
	 * block size includes itself - i.e. is the block size was 2
	 * then there would be no additional bytes in the block.
	 *
	 * So, what we do here is read blocks until we get an SOF
	 * marker, and then extract the width and height from that.
	 */
	for (;;) {
		if (!mustbe(f, 0xff))
			return (0);
		if (!get(f, &ch))
			return (0);
		if (!get(f, &l1))
			return (0);
		if (!get(f, &l2))
			return (0);
		l = (int) l1 *256 + (int) l2;
		/* Is it the block we're looking for? */
		if (ch >= M_SOF0 && ch <= M_SOF3) {
			/* Toss the sample precision. */
			if (!get(f, &ch))
				return (0);
			/* Read the height and width. */
			if (!get(f, &h1))
				return (0);
			if (!get(f, &h2))
				return (0);
			if (!get(f, &w1))
				return (0);
			if (!get(f, &w2))
				return (0);
			w = (int) w1 *256 + (int) w2;
			h = (int) h1 *256 + (int) h2;
			*wp = w;
			*hp = h;
			return (1);
		}
		if ( ch == M_SOS || ch == M_EOI )
			return (0);
		/* Not the block we want.  Read and toss. */
		for (i = 2; i < l; ++i)
			if (!get(f, &ch))
				return (0);
	}
}

static int
get(register FILE * f, register u_char *chP)
{
	register int ich;

	ich = getc(f);
	if (ich == EOF)
		return (0);
	*chP = (u_char) ich;
	return (1);
}


static int
mustbe(register FILE * f, register u_char ch)
{
	u_char ch2;

	if (!get(f, &ch2))
		return (0);
	if (ch2 != ch)
		return (0);
	return (1);
}

static int
png(register FILE * f, register int *wp, register int *hp)
{
	register int l, w, h, d;
	u_char l1, l2, l3, l4, w1, w2, w3, w4, h1, h2, h3, h4, d1;

	if (!mustbe(f, 'P'))
		return (0);
	if (!mustbe(f, 'N'))
		return (0);
	if (!mustbe(f, 'G'))
		return (0);

	if (!mustbe(f, '\r'))
		return (0);
	if (!mustbe(f, '\n'))
		return (0);
	if (!mustbe(f, CTRL('Z')))
		return (0);
	if (!mustbe(f, '\n'))
		return (0);

	/* Skip 4 byte length */
	if (!get(f, &l1) || !get(f, &l2) || !get(f, &l3) || !get(f, &l4))
		return (0);

	l = (int)l1;
	l = (l << 8) + (int)l2;
	l = (l << 8) + (int)l3;
	l = (l << 8) + (int)l4;
	/* IHDR, 4 byte width, 4 byte height and 1 byte depth */
	if (l != 13)
		return (0);

	if (!mustbe(f, 'I'))
		return (0);
	if (!mustbe(f, 'H'))
		return (0);
	if (!mustbe(f, 'D'))
		return (0);
	if (!mustbe(f, 'R'))
		return (0);

	/* Width and height are the next things in the file. */
	if (!get(f, &w1) || !get(f, &w2) || !get(f, &w3) || !get(f, &w4))
		return (0);

	if (!get(f, &h1) || !get(f, &h2) || !get(f, &h3) || !get(f, &h4))
		return (0);

	w = (int)w1;
	w = (w << 8) + (int)w2;
	w = (w << 8) + (int)w3;
	w = (w << 8) + (int)w4;

	h = (int)h1;
	h = (h << 8) + (int)h2;
	h = (h << 8) + (int)h3;
	h = (h << 8) + (int)h4;

	/* Depth */
	get(f, &h1);
	if (!get(f, &d1))
		return (0);
	d = (int) d1;

	*wp = w;
	*hp = h;
	return (1);
}
