OWS     [ \t]*
WS      [ \t]+
NQ      [^"]*
NUM     [0-9]+
IMGNAME [-A-Za-z0-9/._:+]+
TOK     [A-Za-z]+
FODDER  [-A-Za-z0-9/._%]+
IMG     [Ii][Mm][Gg]
SRC     [Ss][Rr][Cc]
WIDTH   [Ww][Ii][Dd][Tt][Hh]
LOWSRC  [Ll][Oo][Ww][Ss][Rr][Cc]
HEIGHT  [Hh][Ee][Ii][Gg][Hh][Tt]

    /*
     * Copyright (c) 1997, 1998, 1999, 2000, 2005, 2009, 2010, 2013, 2021
     * Craig Leres
     * All rights reserved.
     *
     * Redistribution and use in source and binary forms, with or without
     * modification, are permitted provided that the following conditions are
     * met:
     *     * Redistributions of source code must retain the above copyright
     *       notice, this list of conditions and the following disclaimer.
     *     * Redistributions in binary form must reproduce the above copyright
     *       notice, this list of conditions and the following disclaimer in
     *       the documentation and/or other materials provided with the
     *       distribution.
     *     * Neither the name of the University nor the names of its
     *       contributors may be used to endorse or promote products derived
     *       from this software without specific prior written permission.
     *
     * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
     * IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
     * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
     * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR
     * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
     * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
     * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
     * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
     * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
     * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
     * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
     *
     * Read a html file and output it with the width and height tags
     * added or updated.
     */
    #ifndef lint
    static const char copyright[] __attribute__ ((unused)) =
	"@(#) Copyright (c) 1997, 1998, 1999, 2000, 2005, 2009, 2010, 2013, 2021\nCraig Leres\n";
    static const char rcsid[] __attribute__ ((unused)) =
	"@(#) $Id: imgfilt.l 85 2021-12-31 05:35:06Z leres $ (XSE)";
    #endif

    #include <sys/types.h>

    #include <ctype.h>
    #include <errno.h>
    #include <fcntl.h>
    #include <stdio.h>
    #include <string.h>
    #include <unistd.h>

    #include "gnuc.h"
    #ifdef HAVE_OS_PROTO_H
    #include "os-proto.h"
    #endif

    #include "imgfilt.h"
    #include "parseimg.h"
    #include "version.h"

    #define YY_NO_INPUT

    int i, j;
    char *prog;
    char *file = NULL;		/* current input file */
    char *capstr = NULL;	/* pointer to malloced memory */
    int capsize = 0;		/* current size of buffer */
    int caplen = 0;		/* number of bytes in use */
    int width = -1;
    int height = -1;
    int widthpct = 0;		/* is width a percent? */
    int heightpct = 0;		/* is height a percent? */
    int widthind = -1;
    int heightind = -1;
    int lowsrcind = -1;
    int line = 1;		/* current line number */
    int errors;
    static char image[1024];
    static char lowsrc[1024];

    #undef yywrap
    int yywrap(void);
    int yylex(void);
    void regurgitate(void);
    void capture(const char *, int);
    void quotecapture(const char *, int, char *);

%x img
%option nounput

%%


\<{OWS}{IMG}		{
			BEGIN(img);
			caplen = 0;
			width = -1;
			height = -1;
			widthpct = 0;
			heightpct = 0;
			widthind = -1;
			heightind = -1;
			image[0] = '\0';
			lowsrc[0] = '\0';
			lowsrcind = -1;
			capture(yytext, yyleng);
			}

<img>{WS}{WIDTH}{OWS}={OWS}{NUM}% {
			/* Leave percent alone (but add double quotes) */
			++widthpct;
			quotecapture(yytext, yyleng, NULL);
			}

<img>{WS}{HEIGHT}{OWS}={OWS}{NUM}% {
			/* Leave percent alone (but add double quotes) */
			++heightpct;
			quotecapture(yytext, yyleng, NULL);
			}

<img>{WS}{WIDTH}{OWS}={OWS}\"{NUM}%\" {
			/* Leave percent alone */
			++widthpct;
			capture(yytext, yyleng);
			}

<img>{WS}{HEIGHT}{OWS}={OWS}\"{NUM}%\" {
			/* Leave percent alone */
			++heightpct;
			capture(yytext, yyleng);
			}

<img>{WS}{WIDTH}{OWS}={OWS}{NUM}	{
			i = strchr(yytext, '=') - yytext + 1;
			while (isblank(yytext[i]))
			    ++i;
			width = atoi(yytext + i);
			widthind = caplen + i + 1;
			capture(yytext, yyleng);
			}

<img>{WS}{WIDTH}{OWS}={OWS}\"{NUM}\" {
			i = strchr(yytext, '=') - yytext + 1;
			while (isblank(yytext[i]))
			    ++i;
			++i;
			width = atoi(yytext + i);
			widthind = caplen + i + 1;
			capture(yytext, yyleng);
			}


<img>{WS}{HEIGHT}{OWS}={OWS}{NUM}	{
			i = strchr(yytext, '=') - yytext + 1;
			while (isblank(yytext[i]))
			    ++i;
			height = atoi(yytext + i);
			heightind = caplen + i + 1;
			capture(yytext, yyleng);
			}

<img>{WS}{HEIGHT}{OWS}={OWS}\"{NUM}\" {
			i = strchr(yytext, '=') - yytext + 1;
			while (isblank(yytext[i]))
			    ++i;
			++i;
			height= atoi(yytext + i);
			heightind = caplen + i + 1;
			capture(yytext, yyleng);
			}

<img>{WS}{SRC}{OWS}={OWS}\"{NQ}\" {
			i = strchr(yytext, '=') - yytext + 1;
			while (isblank(yytext[i]))
			    ++i;
			++i;
			j = yyleng - i - 1;
			strncpy(image, yytext + i, j);
			image[j] = '\0';
			capture(yytext, yyleng);
			}

<img>{WS}{SRC}{OWS}={OWS}{IMGNAME} {
			quotecapture(yytext, yyleng, image);
			}

<img>{WS}{LOWSRC}{OWS}={OWS}\"{NQ}\" {
			i = strchr(yytext, '=') - yytext + 1;
			while (isblank(yytext[i]))
			    ++i;
			++i;
			lowsrcind = caplen + i;
			quotecapture(yytext, yyleng, lowsrc);
			}

<img>{WS}{LOWSRC}{OWS}={OWS}{IMGNAME} {
			i = strchr(yytext, '=') - yytext + 1;
			while (isblank(yytext[i]))
			    ++i;
			++i;
			lowsrcind = caplen + i + 1;
			j = yyleng - i;
			strncpy(lowsrc, yytext + i, j);
			lowsrc[j] = '\0';
			}

<img>{WS}{TOK}{OWS}={OWS}\"{NQ}\"	capture(yytext, yyleng);
<img>{WS}{TOK}{OWS}={OWS}{FODDER}	capture(yytext, yyleng);
<img>\n			{
			++line;
			capture(yytext, yyleng);
			}

<img>{OWS}\>		{
			if (image[0] == '\0')
				fprintf(stderr,
				    "%s: %s:%d Saw <img> without src\n",
				    prog, file, line);
			capture(yytext, yyleng);
			regurgitate();
			BEGIN(0);
			}

\"{NQ}\"		ECHO;

\n			{
			++line;
			ECHO;
			}


%%
#ifdef DEBUG
int debug = 0;
#endif

int targc;
char **targv;

extern char *optarg;
extern int optind, opterr;

/* Forwards */
void usage(void) __attribute__((noreturn));
int main(int, char **);
void fixuplowsrc(void);

int
main(argc, argv)
	int argc;
	char **argv;
{
	char *cp;
	int op;

	if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	opterr = 0;
	while ((op = getopt(argc, argv, "d")) != EOF)
		switch (op) {

#ifdef DEBUG
		case 'd':
			++debug;
			break;
#endif

		default:
			usage();
		}

	/* Let yywrap() figure out if there are any arguments to open */
	targc = argc - optind;
	targv = &argv[optind];
	yyin = NULL;
	(void)yywrap();

	/* Process file opened by yywrap() or stdin if no arguments */
	if (yyin)
		yylex();

	exit(errors != 0);
}

int
yywrap()
{
	static int didany = 0;

	/* Close file, if necessary */
	if (yyin && yyin != stdin) {
		(void)fclose(yyin);
		yyin = NULL;
	}

	/* Spin through arguments until we run out or successfully open one */
	while (targc > 0) {
		file = targv[0];
		--targc;
		++targv;
		++didany;
		if ((yyin = fopen(file, "r")) != NULL)
			return (0);
		fprintf(stderr, "%s: %s: %s\n", prog, file, strerror(errno));
		++errors;
	}
	if (!didany) {
		file = "<stdin>";
		yyin = stdin;
	}
	return (1);
}

/* Append the next part of the "<img ...>" string to the capture buffer */
void
capture(const char *str, int len)
{

	while (caplen + len + 1 > capsize) {
		if (capsize == 0) {
			capsize = 1024;
			capstr = (char *)malloc(capsize);
		} else {
			capsize <<= 1;
			capstr = (char *)realloc(capstr, capsize);
		}
		if (capstr == NULL) {
			fprintf(stderr, "%s: malloc/realloc: %s\n",
			    prog, strerror(errno));
			exit(1);
		}
	}

	strncpy(capstr + caplen, str, len);
	caplen += len;
	capstr[caplen] = '\0';
}

/* Append the next part of the to the capture buffer, adding double quotes */
void
quotecapture(const char *str, int len, char *meat)
{
	int i, j, k;

	i = strchr(str, '=') - str + 1;
	while (isblank(str[i]))
	    ++i;
	k = i;
	if (str[i] == '"')
		++i;
	j = len - i;
	if (str[i + j - 1] == '"')
		--j;
	if (meat != NULL) {
		strncpy(meat, str + i, j);
		meat[j] = '\0';
	}
	capture(str, k);
	capture("\"", 1);
	capture(str + i, j);
	capture("\"", 1);
}

/* Output the "<img ...>" string, perhaps updating the width and/or height */
void
regurgitate(void)
{
	int w, h;
	int ind1, ind2, val1, val2;
	char *str1, *str2;

	fixuplowsrc();

	errno = 0;
	if (!parseimg(image, &w, &h) ||
	    ((w == width || widthpct) && (h == height || heightpct))) {
		if (errno == ENOENT)
			++errors;
		/*
		 * Couldn't parse it or the width/height are already
		 * correct (perhaps because they are a percent)
		 */
		fprintf(stdout, "%.*s", caplen - 1, capstr);
		fprintf(stdout, ">");
		caplen = 0;
		*capstr = '\0';
		return;
	}

	if (widthind < 0 && !widthpct && heightind < 0 && !heightpct) {
		/* Both are missing */
		fprintf(stdout, "%.*s", caplen - 1, capstr);
		fprintf(stdout, " width=\"%d\" height=\"%d\"", w, h);
		fprintf(stdout, ">");
		caplen = 0;
		*capstr = '\0';
		return;
	}

	if ((widthind < 0 && !widthpct) ||
	    (widthind < heightind && !heightpct)) {
		ind1 = widthind;
		ind2 = heightind;
		val1 = w;
		val2 = h;
		str1 = "width";
		str2 = "height";
	} else {
		ind1 = heightind;
		ind2 = widthind;
		val1 = h;
		val2 = w;
		str1 = "height";
		str2 = "width";
	}

	if (ind1 < 0) {
		fprintf(stdout, "%.*s", ind2 - 1, capstr);
	} else {
		fprintf(stdout, "%.*s", ind1 - 1, capstr);
		while (isdigit(capstr[ind1]))
			++ind1;
		fprintf(stdout, "%d", val1);
		fprintf(stdout, "%.*s", ind2 - ind1 - 1, capstr + ind1);
	}

	if (ind2 < 0) {
		fprintf(stdout, "%.*s", caplen - ind1 - 1, capstr + ind1);
	} else {
		while (isdigit(capstr[ind2]))
			++ind2;
		fprintf(stdout, "%d%.*s",
		    val2, caplen - ind2 - 1, capstr + ind2);
	}

	if (ind1 < 0)
		fprintf(stdout, " %s=\"%d\"", str1, val1);
	if (ind2 < 0)
		fprintf(stdout, " %s=\"%d\"", str2, val2);
	fprintf(stdout, ">");
	caplen = 0;
	*capstr = '\0';
}

void
fixuplowsrc()
{
	int x, y, z;
	char *cp;
	char temp[1024];
	int xerrno;

	/* If we found a lowsrc and it's readable, we're done */
	xerrno = 0;
	if (lowsrcind >= 0) {
		errno = 0;
		if (access(lowsrc, R_OK) >= 0)
			return;
		xerrno = errno;
	}

	/* Try to find it */
	cp = strrchr(image, '.');
	if (cp != NULL) {
		temp[0] = '\0';

		/* webindex style */
		if (strcmp(cp + 1, "jpg") == 0)
			(void)sprintf(temp,
			    ".nai/%.*s__t.jpg", (int)(cp - image), image);
		else
			(void)sprintf(temp, ".nai/%.*s_%s_t.jpg",
			    (int)(cp - image), image, cp + 1);
		if (access(temp, R_OK) < 0)
			temp[0] = '\0';

		if (temp[0] == '\0') {
			(void)sprintf(temp, "%.*s_t.jpg",
			    (int)(cp - image), image);
			if (access(temp, R_OK) < 0)
				temp[0] = '\0';
		}

		if (temp[0] == '\0' && strcmp(cp + 1, "jpg") != 0) {
			(void)sprintf(temp, "%.*s_t.%s",
			    (int)(cp - image), image, cp + 1);
			if (access(temp, R_OK) < 0)
				temp[0] = '\0';
		}
		/* We couldn't find it so just bail */
		if (temp[0] == '\0') {
			/* If there was originally a lowsrc, print an error */
			if (lowsrcind >= 0) {
				fprintf(stderr, "%s: %s: lowsrc %s: %s\n",
				    prog, file, lowsrc, strerror(xerrno));
				    ++errors;
			}
			return;
		}
		(void)strcpy(lowsrc, temp);
	}

	/* Didn't find it */
	if (lowsrc[0] == '\0')
		return;

	/* If we found a thumbnail (and it isn't the source) use it */
	if (lowsrcind < 0) {
		/* Append new lowsrc */
		--caplen;
		cp = " lowsrc=\"";
		capture(cp, strlen(cp));
		capture(lowsrc, strlen(lowsrc));
		cp = "\">";
		capture(cp, strlen(cp));
	} else {
		/* Update lowsrc already in capstr */
		x = strlen(lowsrc);
		y = 0;
		for (cp = capstr + lowsrcind; *cp != '"' && *cp != '\0'; ++cp)
			++y;
		z = x - y;
		if (z > 0 && caplen + z + 1 > capsize) {
			/* We need more space; use clever hack */
			x = caplen;
			capture(capstr, z);
			caplen = x;
		}
		if (z != 0) {
			caplen += z;
			cp = capstr + lowsrcind + y;
			memmove(cp + z, cp, strlen(cp));
		}

		capstr[caplen] = '\0';
		/* Install new lowsrc */
		memmove(capstr + lowsrcind, lowsrc, x);
	}
}

void
usage()
{

	(void)fprintf(stderr, "%s version %s\n", prog, version);
	(void)fprintf(stderr, "usage: %s [-d] [file ...]\n", prog);
	exit(1);
}
