/*
 * @(#) $Id: cf.c 287 2017-01-21 19:15:36Z leres $ (XSE)
 *
 * Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004, 2006, 2007, 2008, 2009, 2015, 2017
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <sys/param.h>
#include <sys/stat.h>

#include <ctype.h>
#include <errno.h>
#include <libgen.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "cf.h"
#include "webindex.h"
#include "util.h"

#define DEF_NAMELEN	13
#define DEF_ICONSIZE	70
#define DEF_IMGBORDER	0
#define DEF_MAXROW	10
#define DEF_MAXCOL	5

/* Forwards */
static char *dstrmerge(char *, char *);
static void freelist(struct list *);
static struct list *makelist(char *);
static int readcf(const char *, struct cf *, int);

/* Open, read and parse one config file */
static int
readcf(const char *fn, struct cf *cf, int mustexist)
{
	int n, d, errs, *ip;
	char *cp, *cp2, *prefixend, **pp;
	struct timespec ts;
	FILE *f;
	struct section *sp;
	struct list **lpp;
	char *cp3, *include, *favicon, *appendhead, *appendtail;
	char buf[1024];
	char at[1024];
	char dname[PATH_MAX];
	char sname[PATH_MAX];
	struct stat sbuf;

	errs = 0;
	f = fopen(fn, "r");
	if (f == NULL) {
		if (mustexist) {
			fprintf(stderr, "%s: readcf: fopen %s/%s: %s\n",
			    prog, prefix, fn, strerror(errno));
			++errs;
		}
		return (errs);
	}

	/* Get the config file mod time */
	if (fstat(fileno(f), &sbuf) >= 0) {
		/* The config file time is the newest of any files included */
		ts = stattim(&sbuf);
		if (tscmp(&cf->c_tim, &ts, <))
			cf->c_tim = ts;
	} else {
		fprintf(stderr, "%s: readcf: fstat %s%s: %s\n",
		    prog, prefix, fn, strerror(errno));
		++errs;
	}

	n = 0;
	while (fgets(buf, sizeof(buf), f) != NULL) {
		++n;

		/* Eat comments */
		cp = strchr(buf, '#');
		if (cp)
			*cp = '\0';

		/* Eat trailing whitesapce */
		cp = buf + strlen(buf) - 1;
		while (cp >= buf && isspace(*cp))
			*cp-- = '\0';
		cp = buf;

		/* Skip blank lines */
		if (*cp == '\n' || *cp == '\0')
			continue;

		/* Get command */
		cp2 = cp;
		while (!isspace(*cp) && *cp != '\0')
			++cp;
		*cp++ = '\0';

		(void)sprintf(at, "%s%s:%d", prefix, fn, n);

		/* Find start of next keyword */
		while (isspace(*cp))
			++cp;

		/* Quoted one line commands */
		include = NULL;
		favicon = NULL;
		if ((pp = &cf->c_datefmt, strcasecmp(cp2, "datefmt") == 0) ||
		    (pp = &favicon, strcasecmp(cp2, "favicon") == 0) ||
		    (pp = &include, strcasecmp(cp2, "include") == 0) ||
		    (pp = &cf->c_indexfn, strcasecmp(cp2, "indexfn") == 0) ||
		    (pp = &cf->c_invisiblepat,
			strcasecmp(cp2, "invisiblepat") == 0) ||
		    (pp = &cf->c_naildir, strcasecmp(cp2, "naildir") == 0) ||
		    (pp = &cf->c_target, strcasecmp(cp2, "target") == 0) ||
		    (pp = &cf->c_tz, strcasecmp(cp2, "tz") == 0)) {
			if (*cp++ != '"') {
				fprintf(stderr,
				    "%s: %s missing leading double quote\n",
				    prog, at);
				++errors;
				continue;
			}
			if (*pp != NULL)
				free(*pp);
			*pp = strdup(cp);
			if (*pp == NULL) {
				fprintf(stderr, "%s: readcf: strdup: %s\n",
				    prog, strerror(errno));
				exit(1);
			}
			cp = *pp;
			cp2 = cp + strlen(cp) - 1;
			if (cp > cp2 || *cp2 != '"') {
				fprintf(stderr,
				    "%s: %s missing trailing double quote\n",
				    prog, at);
				++errors;
				free(*pp);
				*pp = NULL;
				continue;
			}
			*cp2 = '\0';

			/* Process favicon file */
			if (favicon != NULL) {
				strlcpy(dname, prefix, sizeof(dname));
				strlcat(dname, favicon, sizeof(dname));
				/* Collaspe relative file name */
				collapse(dname);
				cf->c_favicon = strdup(dname);
				if (cf->c_favicon == NULL) {
					fprintf(stderr,
					    "%s: readcf: strdup (2x): %s\n",
					    prog, strerror(errno));
					exit(1);
				}
				free(favicon);
				favicon = NULL;
				continue;
			}

			/* Process include file */
			if (include != NULL) {
				strlcpy(dname, dirname(include), sizeof(dname));

				/* Change working directory */
				if (!pushd(dname, &prefixend, &d)) {
					free(include);
					include = NULL;
					continue;
				}
				strlcpy(sname, basename(include),
				    sizeof(sname));
				errs += readcf(sname, cf, 1);
				(void)popd(prefixend, d);
				free(include);
				include = NULL;
				continue;
			}
			continue;
		}

		/* Commands with numeric arguments */
		if ((ip = &cf->c_iconsize, strcasecmp(cp2, "iconsize") == 0) ||
		    (ip = &cf->c_imgborder,
			strcasecmp(cp2, "imgborder") == 0) ||
		    (ip = &cf->c_maxcol, strcasecmp(cp2, "maxcol") == 0) ||
		    (ip = &cf->c_maxrow, strcasecmp(cp2, "maxrow") == 0) ||
		    (ip = &cf->c_namelen, strcasecmp(cp2, "namelen") == 0) ||
		    (ip = &cf->c_tzhours, strcasecmp(cp2, "tzhours") == 0)) {
			*ip = atoi(cp);
			if (*cp == '-' || *cp == '+')
				++cp;
			while (isdigit(*cp))
				++cp;
			while (isspace(*cp))
				++cp;
			if (*cp != '\0') {
				fprintf(stderr,
				    "%s: %s trailing garbage (%s)\n",
				    prog, at, cp);
				++errors;
			}
			continue;
		}

		/* Multi-line commands */
		appendhead = NULL;
		appendtail = NULL;
		if ((pp = &appendhead, strcasecmp(cp2, "appendhead") == 0) ||
		    (pp = &appendtail, strcasecmp(cp2, "appendtail") == 0) ||
		    (pp = &cf->c_body, strcasecmp(cp2, "body") == 0) ||
		    (pp = &cf->c_head, strcasecmp(cp2, "head") == 0) ||
		    (pp = &cf->c_table, strcasecmp(cp2, "table") == 0) ||
		    (pp = &cf->c_tail, strcasecmp(cp2, "tail") == 0) ||
		    (pp = &cf->c_title, strcasecmp(cp2, "title") == 0) ||
		    (pp = &cf->c_tr, strcasecmp(cp2, "tr") == 0)) {
			if (*cp != '\0') {
				fprintf(stderr,
				    "%s: %s trailing garbage (%s)\n",
				    prog, at, cp);
				++errors;
			}
			if (*pp != NULL)
				free(*pp);
			n += suck2dot(f, pp, 1);

			if (appendhead != NULL) {
				cf->c_head = dstrmerge(cf->c_head, appendhead);
				appendhead = NULL;
			}
			if (appendtail != NULL) {
				cf->c_tail = dstrmerge(cf->c_tail, appendtail);
				appendtail = NULL;
			}
			continue;
		}

		/* Lists (these may be redefined) */
		if ((lpp = &cf->c_invisible,
			strcasecmp(cp2, "invisible") == 0)) {
			if (*cp != '\0') {
				fprintf(stderr,
				    "%s: %s trailing garbage (%s)\n",
				    prog, at, cp);
				++errors;
			}
			if (*lpp != NULL)
				freelist(*lpp);
			n += suck2dot(f, &cp3, 0);
			*lpp = makelist(cp3);
			free(cp3);
			continue;
		}

		/* Lists (these may not be redefined) */
		if ((lpp = &cf->c_alttext, strcasecmp(cp2, "alttext") == 0) ||
		    (lpp = &cf->c_captions, strcasecmp(cp2, "captions") == 0) ||
		    (lpp = &cf->c_thumbnails,
			strcasecmp(cp2, "thumbnails") == 0) ||
		    (lpp = &cf->c_timestamps,
			strcasecmp(cp2, "timestamps") == 0)) {
			if (*cp != '\0') {
				fprintf(stderr,
				    "%s: %s trailing garbage (%s)\n",
				    prog, at, cp);
				++errors;
			}
			if (*lpp != NULL) {
				fprintf(stderr,
				    "%s: %s \"%s\" already defined\n",
				    prog, at, cp2);
				exit(1);
			}
			n += suck2dot(f, &cp3, 1);
			*lpp = makelist(cp3);
			free(cp3);
			continue;
		}

		/* Keyword commands */
		if (strcasecmp(cp2, "dirdetail") == 0) {
			cf->c_flags &= ~ C_NO_DIRDETAIL;
			continue;
		} else if (strcasecmp(cp2, "appdirindexfn") == 0) {
			cf->c_flags |= C_APPDIRINDEXFN;
			continue;
		} else if (strcasecmp(cp2, "imgsize") == 0) {
			cf->c_flags &= ~ C_NO_IMGSIZE;
			continue;
		} else if (strcasecmp(cp2, "name") == 0) {
			cf->c_flags &= ~ C_NO_NAME;
			continue;
		} else if (strcasecmp(cp2, "nodirdetail") == 0) {
			cf->c_flags |= C_NO_DIRDETAIL;
			continue;
		} else if (strcasecmp(cp2, "noappdirindexfn") == 0) {
			cf->c_flags &= ~ C_APPDIRINDEXFN;
			continue;
		} else if (strcasecmp(cp2, "noimgsize") == 0) {
			cf->c_flags |= C_NO_IMGSIZE;
			continue;
		} else if (strcasecmp(cp2, "noname") == 0) {
			cf->c_flags |= C_NO_NAME;
			continue;
		} else if (strcasecmp(cp2, "noparent") == 0) {
			cf->c_flags |= C_NO_PARENT;
			continue;
		} else if (strcasecmp(cp2, "nosize") == 0) {
			cf->c_flags |= C_NO_SIZE;
			continue;
		} else if (strcasecmp(cp2, "noupcasesuffix") == 0) {
			cf->c_flags &= ~ C_UPCASESUFFIX;
			continue;
		} else if (strcasecmp(cp2, "parent") == 0) {
			cf->c_flags &= ~ C_NO_PARENT;
			continue;
		} else if (strcasecmp(cp2, "size") == 0) {
			cf->c_flags &= ~ C_NO_SIZE;
			continue;
		} else if (strcasecmp(cp2, "upcasesuffix") == 0) {
			cf->c_flags |= C_UPCASESUFFIX;
			continue;
		}

		/* Sort commands */
		if (strcasecmp(cp2, "sort") == 0) {
			if (strcasecmp(cp, "lexical") == 0)
				cf->c_sort = C_SORT_LEX;
			else if (strcasecmp(cp, "reverse-lexical") == 0)
				cf->c_sort = C_SORT_REVLEX;
			else if (strcasecmp(cp, "timestamp") == 0)
				cf->c_sort = C_SORT_TIME;
			else if (strcasecmp(cp, "reverse-timestamp") == 0)
				cf->c_sort = C_SORT_REVTIME;
			else {
				fprintf(stderr,
				    "%s: %s bad sort type \"%s\"\n",
				    prog, at, cp);
				++errors;
			}
			continue;
		}

		/* Section command */
		if (strcasecmp(cp2, "section") == 0) {
			sp = (struct section *)calloc(1, sizeof(*sp));
			if (sp == NULL) {
				fprintf(stderr, "%s: readcf: calloc: %s\n",
				    prog, strerror(errno));
				exit(1);
			}

			/* XXX put this into a subroutine? */
			pp = &sp->s_file;
			if (*cp++ != '"') {
				fprintf(stderr,
				    "%s: %s missing leading double quote (2)\n",
				    prog, at);
				free(sp);
				++errors;
				continue;
			}
			if (*pp != NULL)
				free(*pp);
			*pp = strdup(cp);
			if (*pp == NULL) {
				fprintf(stderr, "%s: readcf: strdup (2): %s\n",
				    prog, strerror(errno));
				exit(1);
			}
			cp = *pp;
			cp2 = cp + strlen(cp) - 1;
			if (cp > cp2 || *cp2 != '"') {
				fprintf(stderr, "%s: %s missing"
				    " trailing double quote (2)\n", prog, at);
				++errors;
				free(sp);
				free(*pp);
				*pp = NULL;
				continue;
			}
			*cp2 = '\0';

			sp->s_next = cf->c_section;
			cf->c_section = sp;

			n += suck2dot(f, &sp->s_html, 1);
			continue;
		}

		/* Bad commands */
		fprintf(stderr, "%s: %s unknown command \"%s\"\n",
		    prog, at, cp2);
		++errors;
	}
	(void)fclose(f);

	return (errs);
}

struct cf *
parsecf(const char *fn)
{
	struct cf *cf;

	cf = (struct cf *)calloc(1, sizeof(*cf));
	if (cf == NULL) {
		fprintf(stderr, "%s: parsecf: calloc: %s\n",
		    prog, strerror(errno));
		exit(1);
	}
	cf->c_namelen = DEF_NAMELEN;
	cf->c_iconsize = DEF_ICONSIZE;
	cf->c_imgborder = DEF_IMGBORDER;
	cf->c_maxrow = DEF_MAXROW;
	cf->c_maxcol = DEF_MAXCOL;

	errors += readcf(fn, cf, 0);

	/* Sanity checks */
	if (cf->c_namelen < -1) {
		fprintf(stderr,
		    "%s: %s%s invalid namelen (%d)\n",
		    prog, prefix, fn, cf->c_namelen);
		cf->c_namelen = DEF_NAMELEN;
		++errors;
	}
	if (cf->c_iconsize < 10 || cf->c_iconsize > 1024) {
		fprintf(stderr,
		    "%s: %s%s invalid iconsize (%d)\n",
		    prog, prefix, fn, cf->c_iconsize);
		cf->c_iconsize = DEF_ICONSIZE;
		++errors;
	}
	if (cf->c_imgborder < 0 || cf->c_imgborder > 200) {
		fprintf(stderr,
		    "%s: %s%s invalid imgborder (%d)\n",
		    prog, prefix, fn, cf->c_imgborder);
		cf->c_imgborder = DEF_IMGBORDER;
		++errors;
	}
	if (cf->c_maxrow < 1) {
		fprintf(stderr,
		    "%s: %s%s invalid maxrow (%d)\n",
		    prog, prefix, fn, cf->c_maxrow);
		cf->c_maxrow = DEF_MAXROW;
		++errors;
	}
	if (cf->c_maxcol < 1 || cf->c_maxcol > 50) {
		fprintf(stderr,
		    "%s: %s%s invalid maxcol (%d)\n",
		    prog, prefix, fn, cf->c_maxcol);
		cf->c_maxcol = DEF_MAXCOL;
		++errors;
	}
	if (cf->c_tzhours <= -24 || cf->c_tzhours >= 24) {
		fprintf(stderr,
		    "%s: %s%s invalid tzhours (%d)\n",
		    prog, prefix, fn, cf->c_tzhours);
		cf->c_tzhours = 0;
		++errors;
	}
	return (cf);
}

static char *
dstrmerge(char *s1, char *s2)
{
	char *t;

	if (s1 == NULL)
		return (s2);
	if (s2 == NULL)
		return (s1);
	t = malloc(strlen(s1) + strlen(s2) + 1);
	if (t == NULL) {
		fprintf(stderr, "%s: dstrmerge: malloc: %s\n",
		    prog, strerror(errno));
		exit(1);
	}
	strcpy(t, s1);
	strcat(t, s2);
	free(s1);
	free(s2);
	return (t);
}

void
freecf(struct cf *cf)
{
	struct section *sp, *sp2;

	/* Single line options */
	if (cf->c_datefmt != NULL)
		free(cf->c_datefmt);
	if (cf->c_indexfn != NULL)
		free(cf->c_indexfn);
	if (cf->c_invisiblepat != NULL)
		free(cf->c_invisiblepat);
	if (cf->c_naildir != NULL)
		free(cf->c_naildir);
	if (cf->c_favicon != NULL)
		free(cf->c_favicon);
	if (cf->c_target != NULL)
		free(cf->c_target);
	if (cf->c_tz != NULL)
		free(cf->c_tz);

	/* Multi-line options */
	if (cf->c_body != NULL)
		free(cf->c_body);
	if (cf->c_head != NULL)
		free(cf->c_head);
	if (cf->c_table != NULL)
		free(cf->c_table);
	if (cf->c_tail != NULL)
		free(cf->c_tail);
	if (cf->c_title != NULL)
		free(cf->c_title);
	if (cf->c_tr != NULL)
		free(cf->c_tr);

	/* Lists */
	if (cf->c_alttext != NULL)
		freelist(cf->c_alttext);
	if (cf->c_captions != NULL)
		freelist(cf->c_captions);
	if (cf->c_invisible != NULL)
		freelist(cf->c_invisible);
	if (cf->c_thumbnails != NULL)
		freelist(cf->c_thumbnails);
	if (cf->c_timestamps != NULL)
		freelist(cf->c_timestamps);

	/* Section */
	for (sp = cf->c_section; sp != NULL; sp = sp2) {
		sp2 = sp->s_next;
		if (sp->s_file != NULL)
			free(sp->s_file);
		if (sp->s_html != NULL)
			free(sp->s_html);
		free(sp);
	}
	free(cf);
}

static struct list *
makelist(char *str)
{
	int i, n;
	char *cp, *cp2;
	struct list *lp, *list;

	/* Count entries */
	cp = str;
	n = 0;
	do {
		if ((cp = strchr(cp, '\n')) == NULL)
			break;
		++n;
		++cp;
	} while (*cp != '\0');

	/* See if the last one is missing its newline */
	cp = str + strlen(str) - 1;
	if (cp >= str && *cp != '\n')
		++n;

	/* Allocate one extra for the end marker */
	list = (struct list *)calloc(n + 1, sizeof(*lp));
	if (list == NULL) {
		fprintf(stderr, "%s: makelist: calloc: %s\n",
		    prog, strerror(errno));
		exit(1);
	}

	/* Build list */
	lp = list;
	cp = str;

	for (i = 0; i < n; ++i, ++lp) {
		/* Skip leading whitespace */
		while (isspace(*cp) && *cp != '\n')
			++cp;

		/* Out of characters? */
		if (*cp == '\0')
			break;

		/* Empty line? */
		if (*cp == '\n') {
			*cp++ = '\0';
			continue;
		}

		/* Save start of key */
		lp->l_key = cp;

		cp = strchr(cp, '\n');
		/* Missing final newline? */
		if (cp == NULL)
			break;

		/* Terminate value */
		*cp++ = '\0';

		/* Find end of key */
		cp2 = lp->l_key;
		while (!isspace(*cp2) && *cp2 != '\0') {
			/* blackslash escapes next character */
			if (*cp2 == '\\' && cp[1] != '\0')
				memmove(cp2, cp2 + 1, strlen(cp2 + 1) + 1);
			++cp2;
		}

		if (*cp2 == '\0') {
			/* Save start of value */
			lp->l_val = cp2;
		} else {
			/* Terminate key */
			*cp2++ = '\0';

			/* Find start of value */
			while (isspace(*cp2))
				++cp2;

			/* Save start of value */
			lp->l_val = cp2;
		}

		/* Save stuff */
		lp->l_key = strdup(lp->l_key);
		lp->l_val = strdup(lp->l_val);
		if (lp->l_key == NULL || lp->l_val == NULL) {
			fprintf(stderr, "%s: makelist: strdup: %s\n",
			    prog, strerror(errno));
			exit(1);
		}

	}

	return (list);
}

static void
freelist(struct list *lp)
{
	struct list *lp2;

	if (lp == NULL)
		return;
	for (lp2 = lp; lp2->l_key != NULL; ++lp2) {
		free(lp2->l_key);
		free(lp2->l_val);
	}
	free(lp);
}
