#!/bin/sh
# @(#) $Id: jpegrotate.sh 288 2017-07-02 19:51:58Z leres $ (XSE)
#
# jpegrotate - rotate jpeg images
#

prog="`basename $0`"
usage()
{
	echo "${prog} [-n] file ...	# auto-rotate using exitags" 1>&2
	echo "${prog} [-n] -l file ...	# rotate left" 1>&2
	echo "${prog} [-n] -r file ...	# rotate right" 1>&2
	echo "${prog} [-n] -f file ...	# rotate 180 (flip)" 1>&2
	echo "${prog} [-n] -h file ...	# horizontal" 1>&2
	echo "${prog} [-n] -V file ...	# vertical" 1>&2
	echo "${prog} [-n] -u file ...	# undo" 1>&2
	exit 1
}

PATH=/usr/local/bin:${PATH}
export PATH

dryrun=0
option=""
status=0
undo=0
automatic=1

args=`getopt fhlnruV $*`

if [ $? -ne 0 ]; then
	usage
fi

set -- ${args}

for a do
	case "${a}" in

	-l)
		option="-2"
		automatic=0
		shift
		;;

	-r)
		option="-9"
		automatic=0
		shift
		;;

	-f)
		option="-1"
		automatic=0
		shift
		;;

	-h)
		option="-F"
		automatic=0
		shift
		;;

	-n)
		dryrun=1
		shift
		;;

	-V)
		option="-f"
		automatic=0
		shift
		;;

	-u)
		undo=1
		shift
		;;

	--)
		shift
		break
		;;
	esac
done

if [ -n "${option}" -a ${undo} -ne 0 ]; then
	usage
fi

if [ ${automatic} -ne 0 ]; then
	option="-a"
fi

if [ $# -lt 1 ]; then
	usage
fi

# We need the sysv version under Solaris
touch="/usr/bin/touch"
if [ ! -x ${touch} ]; then
	touch="touch"
fi

# Execute (or just echo if -n)
execute() {
	cmd="$*"
	if [ ${dryrun} -eq 0 ]; then
		eval "${cmd}"
		return $?
	fi
	echo "+ ${cmd}"
	return 0
}

if [ ${undo} -ne 0 ]; then
	# Undo logic
	while [ $# -gt 0 ]; do
		r="$1"
		shift

		f="`echo ${r} |
		    sed -e 's/r\.jpg$/.jpg/' -e 's/r\.JPG$/.JPG/'`"
		if [ "${f}" = "${r}" ]; then
			echo "${prog}: ${r} already unrotated? (skipping)"
			status=1
			continue
		fi
		if [ -r ${r} -a -r ${f} ]; then
			echo "${prog}: Both ${r} and ${f} exist (skipping)"
			status=1
			continue
		fi
		if [ -r ${f} ]; then
			echo "${prog}: ${f} already exists (skipping)"
			status=1
			continue
		fi
		if [ ! -r ${r} ]; then
			echo "${prog}: Can't find ${r}"
			status=1
			continue
		fi
		o="${f}-"
		if [ ! -r ${o} ]; then
			echo "${prog}: ${o} missing (skipping)"
			status=1
			continue
		fi

		execute "rm -f ${r}-"
		execute "mv ${r} ${r}-"
		execute "mv ${o} ${f}"
		echo "${prog}: ${r} -> ${f}"
	done
	exit ${status}
fi

# Make sure we have everything we need
for f in exiftags exiftran; do
	which ${f} > /dev/null 2>&1
	if [ $? -ne 0 ]; then
		echo "${prog}: Can't find ${f}" 1>&2
		exit 1
	fi
done

# Rotate logic
while [ $# -gt 0 ]; do
	f="$1"
	shift

	case "`echo ${f} | tr A-Z a-z`" in

	*.jpg)
		# Ok
		;;

	*)
		echo "${prog}: ${f} unexpected file extension (skipping)"
		status=1
		continue
		;;
	esac

	r="`echo ${f} | sed -e 's/r*\(\.[Jj][Pp][Gg]\)$/r\1/'`"
	if [ "${f}" = "${r}" ]; then
		echo "${prog}: ${f} already rotated? (skipping)"
		status=1
		continue
	fi
	if [ -r ${r} ]; then
		echo "${prog}: ${r} already exists, skipping ${f}"
		status=1
		continue
	fi
	if [ ! -r ${f} ]; then
		echo "${prog}: Can't find ${f}"
		status=1
		continue
	fi
	o="${f}-"
	if [ -r ${o} ]; then
		echo "${prog}: ${o} already exists (skipping)"
		status=1
		continue
	fi

	# If we're autodetecting, see if anything needs to be done
	if [ ${automatic} -ne 0 ]; then
		rotate=`exiftags ${f} 2>&1 |
		    awk '
		    /Image Orientation: Top, Left-Hand/ {
			    print "0"
			    next
		    }
		    /Image Orientation: Right-Hand, Top/ {
			    print "90"
			    next
		    }
		    /Image Orientation: Bottom, Right-Hand/ {
			    print "180"
			    next
		    }
		    /Image Orientation: Left-Hand, Bottom/ {
			    print "270"
			    next
		    }
		    /Image Orientation: / {
			    i = index($0, ": ")
			    if (i > 0)
				    print "# " substr($0, i + 2)
			    next
		    }
		    /exiftags: maker .* not supported/ {
			    # ignore
			    next
		    }
		    /exiftags: / {
			    # Print errors
			    print
		    }
		    '`
		case "${rotate}" in

		0)
			# No transformation needed
			continue
			;;

		[0-9]*)
			# Ok!
			;;

		"")
			echo "${prog}: Can't parse ${f} (skipping)" 1>&2
			continue
			;;

		*)
			echo "${prog}: Can't parse ${f} (rotate: ${rotate}, skipping)" 1>&2
			continue
			;;
		esac
	fi

	# Pick up permissions from original file
	execute "cp -p ${f} ${r}"
	if [ $? -ne 0 ]; then
		exit 1
	fi
	execute "chmod +w ${r}"

	# Transform
	execute "exiftran ${option} -o ${r} ${f}"
	if [ $? -ne 0 ]; then
		rm ${r}
		exit 1
	fi

	# Do this last in case exiftran fails
	execute "mv ${f} ${o}"

	# Copy access and modified timestamps
	execute "${touch} -r ${o} ${r}"
	echo "${prog}: ${f} -> ${r}"
done

exit ${status}
