/*
 * @(#) $Id: icon.c 267 2015-07-04 21:54:44Z leres $ (XSE)
 *
 * Copyright (c) 1999, 2000, 2001, 2006, 2008, 2009, 2010, 2013, 2015
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "cf.h"
#include "webindex.h"
#include "util.h"

#include "gif.h"
#include "icon.h"

/* Locals */
static const u_char avibytes[] = {
#include "aviicon.c"
};

static const u_char dirbytes[] = {
#include "diricon.c"
};

static const u_char filebytes[] = {
#include "fileicon.c"
};

static const u_char flashbytes[] = {
#include "flashicon.c"
};

static const u_char mp3bytes[] = {
#include "mp3icon.c"
};

static const u_char parentbytes[] = {
#include "parenticon.c"
};

static const u_char pdfbytes[] = {
#include "pdficon.c"
};

static const u_char qtbytes[] = {
#include "qticon.c"
};

static const u_char realbytes[] = {
#include "realicon.c"
};

static const char aviname[] =	".avi.gif";
static const char dirname[] =	".dir.gif";
static const char filename[] =	".file.gif";
static const char flashname[] =	".flash.gif";
static const char mp3name[] =	".mp3.gif";
static const char parentname[] = ".parent.gif";
static const char pdfname[] =	".pdf.gif";
static const char qtname[] =	".qt.gif";
static const char realname[] =	".real.gif";


/* Globals */
struct s2v str2icon[] = {
	{ aviname,	C_AVI_ICON },
	{ dirname,	C_DIR_ICON },
	{ filename,	C_FILE_ICON },
	{ flashname,	C_FLASH_ICON },
	{ mp3name,	C_MP3_ICON },
	{ parentname,	C_PARENT_ICON },
	{ pdfname,	C_PDF_ICON },
	{ qtname,	C_QT_ICON },
	{ realname,	C_REAL_ICON },
	{ NULL,		0 }				/* Default value */
};

struct v2v wtyp2bit[] = {
	{ T_AVI,	C_AVI_ICON },
	{ T_DIR,	C_DIR_ICON },
	{ T_FILE,	C_FILE_ICON },
	{ T_FLASH,	C_FLASH_ICON },
	{ T_HTML,	C_FILE_ICON },
	{ T_MP3,	C_MP3_ICON },
	{ T_PARENT,	C_PARENT_ICON },
	{ T_PDF,	C_PDF_ICON },
	{ T_QT,		C_QT_ICON },
	{ T_REAL,	C_REAL_ICON },
	{ 0,		0 }				/* Default value */
};

/* Create the desired icon (but only if needed) */
const char *
createicon(int t, struct cf *cf, int *wp, int *hp)
{
	const u_char *code;
	size_t n, cc;
	int fd;
	const char *name;
	struct gifheader *gp;
	static char file[1024];

	switch (t) {

	case T_AVI:
		name = aviname;
		code = avibytes;
		n = sizeof(avibytes);
		break;

	case T_DIR:
		name = dirname;
		code = dirbytes;
		n = sizeof(dirbytes);
		break;

	case T_FILE:
	case T_HTML:
		name = filename;
		code = filebytes;
		n = sizeof(filebytes);
		break;

	case T_FLASH:
		name = flashname;
		code = flashbytes;
		n = sizeof(flashbytes);
		break;

	case T_MP3:
		name = mp3name;
		code = mp3bytes;
		n = sizeof(mp3bytes);
		break;

	case T_PARENT:
		name = parentname;
		code = parentbytes;
		n = sizeof(parentbytes);
		break;

	case T_PDF:
		name = pdfname;
		code = pdfbytes;
		n = sizeof(pdfbytes);
		break;

	case T_REAL:
		name = realname;
		code = realbytes;
		n = sizeof(realbytes);
		break;

	case T_QT:
		name = qtname;
		code = qtbytes;
		n = sizeof(qtbytes);
		break;

	default:
		fprintf(stderr, "%s: createicon: unknown type %d\n", prog, t);
		exit(1);
	}

	/* Extract gif dimensions */
	gp = (struct gifheader *)code;
	*wp = GIF_WIDTH(gp);
	*hp = GIF_HEIGHT(gp);

	/* Only create icon if it doesn't already exist */
	(void)sprintf(file, "%s/%s", naildir(cf), name);
	if (access(file, F_OK) < 0) {
		if (verbose)
			fprintf(stderr, "%s: creating %s%s\n",
			    prog, prefix, file);
		fd = creat(file, 0666);
		if (fd < 0) {
			fprintf(stderr, "%s: createicon: creat %s: %s%s\n",
			    prog, prefix, file, strerror(errno));
			++errors;
		}
		cc = write(fd, code, n);
		if ((int)cc < 0) {
			fprintf(stderr, "%s: createicon: write %s: %s%s\n",
			    prog, prefix, file, strerror(errno));
			++errors;
		}
		if (cc != n) {
			fprintf(stderr,
			    "%s: createicon: short write %s: %d != %d\n",
			    prog, file, (int)cc, (int)n);
			++errors;
		}
		if (close(fd) < 0) {
			fprintf(stderr, "%s: createicon: close %s: %s%s\n",
			    prog, prefix, file, strerror(errno));
			++errors;
		}
	}

	return (file);
}
