/*
 * @(#) $Id: util.c 274 2015-07-05 22:01:31Z leres $ (XSE)
 *
 * Copyright (c) 1999, 2000, 2001, 2002, 2006, 2008, 2009, 2010, 2015
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <sys/types.h>

#include <ctype.h>
#include <errno.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "cf.h"
#include "webindex.h"
#include "util.h"

const char dotdot[] = "../";
#define DOTDOT_LEN (sizeof(dotdot) - 1)
const char *dot = dotdot + 1;
#define DOT_LEN (DOTDOT_LEN - 1)

/*
 * Inline collapse relative file path
 *
 * Assume the new shorter string will always fits
 */
void
collapse(char *p)
{
	char *p2, *p3;

	/* Remove leading "./" and step over leading "../" strings  */
	while (*p != '\0') {
		if (strncmp(p, dot, DOT_LEN) == 0) {
			/* Remove */
			memmove(p, p + DOT_LEN, strlen(p + DOT_LEN) + 1);
			continue;
		}
		if (strncmp(p, dotdot, DOTDOT_LEN) == 0) {
			/* Keep */
			p += DOTDOT_LEN;
			continue;
		}
		break;
	}

	/* Remove "../" and the token before */
	while ((p2 = strstr(p, dotdot)) != NULL) {
		/* Find previous '/' */
		p3 = p2 - 1;
		for (;;) {
			/* Lose */
			if (p3 < p) {
				p = p2 + DOTDOT_LEN;
				break;
			}
			if (p == p3 || p3[-1] == '/') {
				memmove(p3, p2 + DOTDOT_LEN,
				    strlen(p2 + DOTDOT_LEN) + 1);
				break;
			}
			--p3;
		}
	}
}

const char *
escapename(const char *name)
{
	char *cp;
	const char *p;
	size_t n;
	static char *result = NULL;

	if (result != NULL) {
		free(result);
		result = NULL;
	}
	for (n = 0, p = name; *p != '\0'; ++p)
		if (*p == '\'')
			++n;
	/* We need 3 extra characters per escape */
	n *= 3;

	/* Add space for leading and trailing quotes and EOS */
	result = malloc(strlen(name) + n + 3);
	if (result == NULL) {
		fprintf(stderr, "%s: escapenaem: malloc: %s",
		    prog, strerror(errno));
		exit(1);
	}
	p = name;
	cp = result;
	*cp++ = '\'';
	while (*p != '\0') {
		if (*p == '\'') {
			*cp++ = '\'';
			*cp++ = '\\';
			*cp++ = '\'';
		}
		*cp++ = *p++;
	}
	*cp++ = '\'';
	*cp = '\0';

	return (result);
}

static const char token[] = "%PREFIX%";
#define TOKEN_LEN (sizeof(token) - 1)

void
expandprefix(char *dst, const char *src, size_t size)
{
	int n;
	const char *p;

	for (;;) {
		p = strstr(src, token);
		if (p == NULL) {
			/* No match; copy the rest */
			strlcpy(dst, src, size);
			return;
		}

		/* Copy the first part */
		n = p - src + 1;
		if (n > size)
			n = size;
		strlcpy(dst, src, n);
		--n;
		dst += n;
		src += n;
		size -= n;

		/* Copy the prefix */
		strlcpy(dst, prefix, size);
		n = strlen(prefix);
		dst += n;
		size -= n;
		src = p + TOKEN_LEN;
	}
}

char *
getlistval(struct list *lp, const char *name, int isdir)
{
	if (lp != NULL)
		for (; lp->l_key != NULL; ++lp)
			if (matchfileordir(name, lp->l_key, isdir)) {
				lp->l_flags |= L_REFERENCED;
				return (lp->l_val);
			}
	return (NULL);
}

int
matchfileordir(const char *name1, const char *name2, int isdir)
{
	int l1, l2;

	if (!isdir)
		return (strcmp(name1, name2) == 0);

	l1 = strlen(name1);
	if (l1 > 0 && name1[l1 - 1] == '/')
		--l1;
	l2 = strlen(name2);
	if (l2 > 0 && name2[l2 - 1] == '/')
		--l2;
	if (l1 != l2)
		return (0);
	return (strncmp(name1, name2, l1) == 0);
}

void
reportunused(struct list *lp, const char *what)
{
	if (lp != NULL)
		for (; lp->l_key != NULL; ++lp)
			if ((lp->l_flags & L_REFERENCED) == 0)
				fprintf(stderr, "%s: warning: %s%s: %s\n",
				    prog, prefix, lp->l_key, what);
}

#ifndef HAVE_SETENV
int
setenv(const char *n, const char *v, int overwrite)
{
	static char e[256];

	(void)snprintf(e, sizeof(e), "%s=%s", n, v);
	return (putenv(e));
}
#endif

const char *
simplewd(void)
{
	char *cp;
	static char wd[1024];

	if (getcwd(wd, sizeof(wd)) == NULL) {
		fprintf(stderr, "%s: simplewd: getcwd: %s\n",
		    prog, strerror(errno));
		exit(1);
	}
	cp = strrchr(wd, '/');
	if (cp == NULL)
		return (wd);
	return (++cp);
}

int
str2val(const struct s2v *lp, const char *suffix)
{
	const struct s2v *tp;

	for (tp = lp; tp->s != NULL; ++tp)
		if (strcmp(suffix, tp->s) == 0)
			break;

	return (tp->v);
}

int
str2valcase(const struct s2v *lp, const char *suffix)
{
	const struct s2v *tp;

	for (tp = lp; tp->s != NULL; ++tp)
		if (strcasecmp(suffix, tp->s) == 0)
			break;

	return (tp->v);
}

int
suck2dot(FILE *f, char **pp, int doprefix)
{
	int n, size, len, i;
	char *p, *cp;
	char buf[2048];
	char buf2[2048];

	n = 0;
	size = 256;
	len = 0;
	p = (char *)malloc(size);
	if (p == NULL) {
		fprintf(stderr, "%s: suck2dot: malloc: %s",
		    prog, strerror(errno));
		exit(1);
	}
	*p = '\0';
	*pp = p;
	while (fgets(buf, sizeof(buf), f) != NULL) {
		++n;
		cp = buf;
		/* Eat leading dot; done if there was only one */
		if (*cp == '.' && *++cp == '\n' && cp[1] == '\0')
			break;

		if (doprefix) {
			expandprefix(buf2, buf, sizeof(buf2));
			cp = buf2;
		}
		i = strlen(cp);
		if (len + i + 1 > size) {
			size += i + 256;
			*pp = (char *)realloc(*pp, size);
			if (*pp == NULL) {
				fprintf(stderr, "%s: suck2dot: realloc: %s",
				    prog, strerror(errno));
				exit(1);
			}
			p = *pp + len;
		}
		strcpy(p, cp);
		p += i;
		len += i;
	}
	return (n);
}

int
typ2flag(const struct v2v *lp, int wtype)
{
	const struct v2v *vp;

	for (vp = lp; vp->v1 != 0; ++vp)
		if (vp->v1 == wtype)
			break;
	return (vp->v2);
}
