/*
 * @(#) $Id: exif.c 339 2023-09-27 06:53:05Z leres $ (XSE)
 *
 * Copyright (c) 2023
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>

#include <libexif/exif-data.h>
#include <libexif/exif-loader.h>

#include "exif.h"

/* Forwards */
static void getgpstag(ExifData *, ExifTag, char *, size_t);
static void massagegps(char *, size_t, const char *);
static ExifData *my_exif_data_new_from_f(FILE *);

static void
getgpstag(ExifData *ed, ExifTag tag, char *buf, size_t len)
{
	ExifEntry *e;

	e = exif_content_get_entry(ed->ifd[EXIF_IFD_GPS], tag);
	buf[0] = '\0';
	exif_entry_get_value(e, buf, len);
}

static void
massagegps(char *buf, size_t len, const char *ref)
{

	long lv;
	float fv, degree;
	char *cp, *ep;

	if (buf[0] == '\0')
		return;

	/* Clobber unprintable */
	for (cp = buf; *cp != '\0'; ++cp)
		if (!isprint(*cp))
			*cp = ' ';

	cp =  buf;

	lv = strtol(cp, &ep, 10);
	if (*ep != ',')
		return;
	degree = (float)lv;
	cp = ep + 1;
	if (*cp != ' ')
		return;
	++cp;
	while (*cp == ' ')
		++cp;

	fv = strtof(cp, &ep);
	if (*ep != ',')
		return;
	degree += fv / 60.0;
	cp = ep + 1;
	if (*cp != ' ')
		return;
	++cp;
	while (*cp == ' ')
		++cp;

	fv = strtof(cp, &ep);
	if (*ep != '\0')
		return;
	degree += fv / 3600.0;

	if (ref[0] == 'S' || ref[0] == 'W')
		degree = -degree;

	(void)snprintf(buf, len, "%.06f", degree);
}

static ExifData *
my_exif_data_new_from_f(FILE *f)
{
	off_t off;
	size_t len;
	ExifData *edata;
	ExifLoader *loader;
	unsigned char data[1024];

	loader = exif_loader_new();
	off = ftello(f);
	rewind(f);
	for (;;) {
		len = fread(data, 1, sizeof(data), f);
		if (len <= 0)
			break;
		if (!exif_loader_write(loader, data, len))
			break;
	}
	(void)fseeko(f, off, SEEK_SET);
	edata = exif_loader_get_data(loader);
	exif_loader_unref(loader);
	return (edata);
}

int
parseexif(FILE *f, char *buf, size_t len)
{
	ExifData *ed;
	char latref[4];
	char lat[32];
	char lonref[4];
	char lon[32];

	buf[0] = '\0';
	ed = my_exif_data_new_from_f(f);
	if (ed != NULL) {
		getgpstag(ed, EXIF_TAG_GPS_LATITUDE_REF,
		    latref, sizeof(latref));
		getgpstag(ed, EXIF_TAG_GPS_LATITUDE, lat, sizeof(lat));
		massagegps(lat, sizeof(lat), latref);
		getgpstag(ed, EXIF_TAG_GPS_LONGITUDE_REF,
		    lonref, sizeof(lonref));
		getgpstag(ed, EXIF_TAG_GPS_LONGITUDE, lon, sizeof(lon));
		massagegps(lon, sizeof(lon), lonref);
		if (latref[0] != '\0' && lat[0] != '\0' &&
		    lonref[0] != '\0' && lon[0] != '\0') {
		    (void)snprintf(buf, len, "%s, %s", lat, lon);
		}
		exif_data_unref(ed);
	}
	return (0);
}
