/*
 * @(#) $Id: exif.c 344 2023-10-08 02:12:27Z leres $ (XSE)
 *
 * Copyright (c) 2023
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <ctype.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libexif/exif-data.h>
#include <libexif/exif-loader.h>
#include <libexif/exif-ifd.h>

#include "exif.h"

/* Forwards */
static time_t getexiftime(ExifData *);
static time_t getexiftime2(ExifData *, const char *);
static time_t getgstime(ExifData *);
static void gettag(ExifData *, int, ExifTag, char *, size_t);
static void massagegps(char *, size_t, const char *);
static ExifData *my_exif_data_new_from_f(FILE *);

void
exiflocation(FILE *f, char *buf, size_t len)
{
	ExifData *ed;
	char latref[4];
	char lat[32];
	char lonref[4];
	char lon[32];

	buf[0] = '\0';
	ed = my_exif_data_new_from_f(f);
	if (ed != NULL) {
		gettag(ed, EXIF_IFD_GPS, EXIF_TAG_GPS_LATITUDE_REF,
		    latref, sizeof(latref));
		gettag(ed, EXIF_IFD_GPS, EXIF_TAG_GPS_LATITUDE,
		    lat, sizeof(lat));
		massagegps(lat, sizeof(lat), latref);
		gettag(ed, EXIF_IFD_GPS, EXIF_TAG_GPS_LONGITUDE_REF,
		    lonref, sizeof(lonref));
		gettag(ed, EXIF_IFD_GPS, EXIF_TAG_GPS_LONGITUDE,
		    lon, sizeof(lon));
		massagegps(lon, sizeof(lon), lonref);
		if (latref[0] != '\0' && lat[0] != '\0' &&
		    lonref[0] != '\0' && lon[0] != '\0') {
		    (void)snprintf(buf, len, "%s, %s", lat, lon);
		}
		exif_data_unref(ed);
	}
}

static time_t
getexiftime(ExifData *ed)
{
	char *cp;
	struct tm tm;
	char offset[16];
	char datetime[32];
	char datetime2[64];

	memset(datetime, 0, sizeof(datetime));
	gettag(ed, EXIF_IFD_GPS, EXIF_TAG_DATE_TIME,
	    datetime, sizeof(datetime));
	if (datetime[0] == '\0')
		return (0);

	memset(offset, 0, sizeof(offset));
	gettag(ed, EXIF_IFD_GPS, EXIF_TAG_OFFSET_TIME, offset, sizeof(offset));
	if (offset[0] == '\0')
		return (0);
	cp = strchr(offset, ':');
	if (cp != NULL)
		strcpy(cp, cp + 1);

	snprintf(datetime2, sizeof(datetime2), "%s %s", datetime, offset);

	cp = strptime(datetime2, "%Y:%m:%d %H:%M:%S %z", &tm);
	if (*cp != '\0')
		return (0);

	return (mktime(&tm));
}

static time_t
getexiftime2(ExifData *ed, const char *tz)
{
	char *cp, *datetime3, *tzfmt;
	struct tm tm;
	char datetime[32];
	char datetime2[64];

	memset(datetime, 0, sizeof(datetime));
	gettag(ed, EXIF_IFD_EXIF, EXIF_TAG_DATE_TIME_ORIGINAL,
	    datetime, sizeof(datetime));
	if (datetime[0] == '\0')
		return (0);

	if (tz == NULL) {
		/* Use system default tz */
		datetime3 = datetime;
		tzfmt = "";
		tzfmt = "%Y:%m:%d %H:%M:%S";
	} else {
		/* Use specified tz */
		snprintf(datetime2, sizeof(datetime2), "%s %s", datetime, tz);
		datetime3 = datetime2;
		tzfmt = "%Y:%m:%d %H:%M:%S %z";
	}

	cp = strptime(datetime3, tzfmt, &tm);
	if (*cp != '\0')
		return (0);

	return (mktime(&tm));
}

static time_t
getgstime(ExifData *ed)
{
	char *cp;
	struct tm tm;
	char date[32];
	char time[32];
	char datetime[32];

	memset(date, 0, sizeof(date));
	gettag(ed, -1, EXIF_TAG_GPS_DATE_STAMP, date, sizeof(date));
	if (date[0] == '\0')
		return (0);

	memset(time, 0, sizeof(time));
	gettag(ed, -1, EXIF_TAG_GPS_TIME_STAMP, time, sizeof(time));
	if (time[0] == '\0')
		return (0);
	cp = strchr(time, '.');
	if (cp != NULL)
		*cp = '\0';

	(void)snprintf(datetime, sizeof(datetime), "%s %s -0000", date, time);
	cp = strptime(datetime, "%Y:%m:%d %H:%M:%S %z", &tm);
	if (cp == NULL || *cp != '\0')
		return (0);

	return (mktime(&tm));
}

time_t
exiftime(FILE *f, const char *tz)
{
	ExifData *ed;
	time_t t;

	ed = my_exif_data_new_from_f(f);
	if (ed == NULL)
		return (0);

	t = getexiftime(ed);
	if (t != 0)
		return (t);

	t = getgstime(ed);
	if (t != 0)
		return (t);

	t = getexiftime2(ed, tz);
	if (t != 0)
		return (t);

	return (0);
}

static void
gettag(ExifData *ed, int ifd, ExifTag tag, char *buf, size_t len)
{
	int i;
	ExifEntry *e;

	if (ifd >= 0) {
		e = exif_content_get_entry(ed->ifd[ifd], tag);
	} else {
		for (i = 0; i < EXIF_IFD_COUNT; ++i) {
			e = exif_content_get_entry(ed->ifd[i], tag);
			if (e != NULL) {
// fprintf(stderr, "gettag %d 0x%x\n", i, tag);
				break;
			}
		}
	}
	buf[0] = '\0';
	if (e != NULL)
		(void)exif_entry_get_value(e, buf, len);
}

static void
massagegps(char *buf, size_t len, const char *ref)
{

	long lv;
	float fv, degree;
	char *cp, *ep;

	if (buf[0] == '\0')
		return;

	/* Clobber unprintable */
	for (cp = buf; *cp != '\0'; ++cp)
		if (!isprint(*cp))
			*cp = ' ';

	cp =  buf;

	lv = strtol(cp, &ep, 10);
	if (*ep != ',')
		return;
	degree = (float)lv;
	cp = ep + 1;
	if (*cp != ' ')
		return;
	++cp;
	while (*cp == ' ')
		++cp;

	fv = strtof(cp, &ep);
	if (*ep != ',')
		return;
	degree += fv / 60.0;
	cp = ep + 1;
	if (*cp != ' ')
		return;
	++cp;
	while (*cp == ' ')
		++cp;

	fv = strtof(cp, &ep);
	if (*ep != '\0')
		return;
	degree += fv / 3600.0;

	if (ref[0] == 'S' || ref[0] == 'W')
		degree = -degree;

	(void)snprintf(buf, len, "%.06f", degree);
}

static ExifData *
my_exif_data_new_from_f(FILE *f)
{
	off_t off;
	size_t len;
	ExifData *edata;
	ExifLoader *loader;
	unsigned char data[1024];

	loader = exif_loader_new();
	off = ftello(f);
	rewind(f);
	for (;;) {
		len = fread(data, 1, sizeof(data), f);
		if (len <= 0)
			break;
		if (!exif_loader_write(loader, data, len))
			break;
	}
	(void)fseeko(f, off, SEEK_SET);
	edata = exif_loader_get_data(loader);
	exif_loader_unref(loader);
	return (edata);
}
