/*
 * @(#) $Id: jpegtagtime.c 344 2023-10-08 02:12:27Z leres $ (XSE)
 *
 * Copyright (c) 2023
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */

#include <sys/types.h>
#include <sys/stat.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "exif.h"

/* Globals */
const char *prog;
int debug;
int force;
int test;
const char *tz;

extern char *optarg;
extern int optind, opterr, optopt;

/* Forwards */
int main(int, char **);
int process(const char *);
void usage(void) __attribute__((noreturn));

int
main(int argc, char **argv)
{
	int i, op, ret;
	char *cp;

	if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	while ((op = getopt(argc, argv, "dfgnT:")) != EOF)
		switch (op) {

		case 'd':
			++debug;
			break;

		case 'f':
			++force;
			break;

		case 'g':
			tz = "-0000";
			break;

		case 'n':
			++test;
			break;

		case 'T':
			tz = optarg;
			break;

		default:
			usage();
		}

	if (argc < optind + 1)
		usage();

	ret = 0;
	for (i = optind; i < argc; ++i)
		if (!process(argv[i]))
			ret = 1;

	exit(ret);
}

/* Returns 1 when successful */
int
process(const char *name)
{
	FILE *f;
	double dv;
	time_t et, mt;
	long offset;
	struct tm *tp;
	struct stat sbuf;
	struct timeval times[2];
	char buf[64];

	f = fopen(name, "r");
	if (f == NULL) {
		fprintf(stderr, "%s: open %s: %s\n",
		    prog, name, strerror(errno));
		return (0);
	}

	if (fstat(fileno(f), &sbuf) < 0) {
		fprintf(stderr, "%s: fstat %s: %s\n",
		    prog, name, strerror(errno));
		fclose(f);
		return(0);
	}
	mt = sbuf.st_mtime;

	et = exiftime(f, tz);

	/* Ignore */
	if (et == 0) {
		fprintf(stderr, "%s: %s missing exif tags\n", prog, name);
		fclose(f);
		return (1);
	}

	dv = difftime(mt, et);
	offset = (time_t)dv;

	/* No change */
	if (offset == 0) {
		fclose(f);
		return (1);
	}

	if (debug)
		fprintf(stderr, "# %s offset %ld\n", name, offset);

	if (test) {
		fclose(f);
		return (1);
	}

	memset(times, 0, sizeof(times));
	times[0].tv_sec = sbuf.st_atime;
	times[1].tv_sec = et;
	if (futimes(fileno(f), times) < 0) {
		fprintf(stderr, "%s: futimes %s: %s\n",
		    prog, name, strerror(errno));
		fclose(f);
		return (0);
	}
	fclose(f);

	tp = localtime(&et);
	(void)strftime(buf, sizeof(buf), "%Y-%m-%d %H:%M:%S %z", tp);

	printf("%s %s\n", buf, name);

	return (1);
}

void
usage(void)
{
	fprintf(stderr, "usage: %s [-dfgn] [-T TZ] FILE ...\n", prog);
	exit(1);
}
