/*
 * @(#) $Id: webindex.c 355 2025-05-30 23:19:48Z leres $ (XSE)
 *
 * Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2015, 2016, 2017, 2018, 2019, 2022, 2023, 2025
 *	Craig Leres
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code
 * distributions retain the above copyright notice and this paragraph
 * in its entirety, and (2) distributions including binary code include
 * the above copyright notice and this paragraph in its entirety in
 * the documentation or other materials provided with the distribution
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
static const char copyright[] =
    "@(#) Copyright (c) 1999, 2000, 2001, 2002, 2003, 2004, 2005, 2006, 2007, 2008, 2009, 2010, 2013, 2015, 2016, 2017, 2018, 2019, 2022, 2023, 2025\nCraig Leres\n";
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#ifdef HAVE_MALLOC_H
#include <malloc.h>
#endif
#ifdef HAVE_PATHS_H
#include <paths.h>
#endif
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>

#ifdef HAVE_DIRENT_H
#include <dirent.h>
#define DIRENTRY struct dirent
#endif
#ifdef HAVE_SYS_DIR_H
#define DIRENTRY struct direct
#include <sys/dir.h>
#endif

#include "cf.h"
#include "webindex.h"
#include "util.h"
#include "match.h"
#include "bmp.h"
#include "icon.h"
#include "jpeg.h"
#include "gif.h"
#include "png.h"
#include "setsignal.h"
#ifdef HAVE_LIBTIFF
#include "tiff.h"
#endif
#include "xbm.h"
#include "xwd.h"
#ifdef HAVE_LIBPPM
#include "wpbm.h"
#endif

/* Convert value to K */
#define C2K(k) (((k) + 1023) / 1024)

/* Is this webfile a directory? */
#define WEBFILE_ISDIR(wp) ((wp)->w_type == T_DIR || (wp)->w_type == T_PARENT)
#define WEBFILE_ISINVISIBLE(wp) (((wp)->w_flags & W_INVISIBLE) != 0)
#define WEBFILE_ISREFERENCED(wp) (((wp)->w_flags & W_REFERENCED) != 0)

#define PLURAL(n) ((n) == 1 ? "" : "s")

/* Info we need to display one file */
struct webfile {
	int w_type;			/* T_FILE, etc. */
	int w_flags;
	int w_kbytes;			/* size in Kbytes */
	int w_nfiles;			/* number of files if a T_DIR */
	struct timespec w_tim;		/* image mod/inode timespec */
	struct timespec w_tn_tim;	/* thumbnail mod timespec */
	time_t w_timestamp;		/* configured picture timestamp */
	int w_w;			/* image width */
	int w_h;			/* image height */
	int w_tnw;			/* thumbnail width */
	int w_tnh;			/* thumbnail height */
	struct webfile *w_tn;		/* configured thumbnail */
	mode_t w_mode;			/* file mode from stat() */
	char w_name[1024];		/* simple filename */
	char w_location [32];		/* GPS location */
};

/* Flags */
#define W_HAVETN	0x1		/* we know how to generate thumbnail */
#define W_INVISIBLE	0x2		/* do not include in index.html */
#define W_REFERENCED	0x4		/* referenced as a configured tn */

/* Globals */
const char *prog;
int fflag;
int recurse;
int killed;				/* child was killed */
char *reason;

const char defindexfn[] = "index.html";
const char defdatefmt[] = "%l:%M%p";

extern char *optarg;
extern int optind, opterr, optopt;

/* Locals */
static const char tz[] = "TZ";
static const char *currenttz;
static const char magic[] = "This file produced by webindex; do not edit";
static const char tnlowsuffix[] = "_t.jpg";
static const char tnuppersuffix[] = "_t.JPG";
static int tnlowsuffixlen = sizeof(tnlowsuffix) - 1;
static int tnuppersuffixlen = sizeof(tnuppersuffix) - 1;
static const char gmapsfmt[] = "https://www.google.com/maps/place/%s";
static const char stylefmt[] =
	"<style>\n"
	"table.w tr, table.w td {\n"
	"    text-align: center;\n"
	"    vertical-align: top;\n"
	"%s}\n"
	/* https://stackoverflow.com/a/65083940 */
	".location {\n"
	"    box-sizing: border-box;\n"
	"    position: relative;\n"
	"    display: inline-block;\n"
	"    width: 16px;\n"
	"    height: 16px;\n"
	"    border: 5px solid blue;\n"
	"    border-radius: 100%%;\n"
	"}\n"
	".location::after {\n"
	"    position: absolute;\n"
	"    top: 100%%;\n"
	"    left: 50%%;\n"
	"    margin: 1px 0 0 -6px;\n"
	"    display: block;\n"
	"    content: '';\n"
	"    border: 6px solid transparent;\n"
	"    border-top: 10px solid blue;\n"
	"    border-bottom: none;\n"
	"}\n"
	"</style>\n";

static struct s2v suffix2type[] = {
	{ "avi",	T_AVI },
	{ "bmp",	T_BMP },
	{ "gif",	T_GIF },
	{ "html",	T_HTML },
	{ "jpeg",	T_JPEG },
	{ "jpg",	T_JPEG },
	{ "mov",	T_QT },
	{ "mp3",	T_MP3 },
	{ "pbm",	T_PBM },
	{ "pdf",	T_PDF },
	{ "pgm",	T_PBM },
	{ "png",	T_PNG },
	{ "pnm",	T_PBM },
	{ "ppm",	T_PBM },
	{ "ra",		T_REAL },
	{ "ram",	T_REAL },
	{ "rm",		T_REAL },
	{ "swf",	T_FLASH },
	{ "tif",	T_TIFF },
	{ "tiff",	T_TIFF },
	{ "xbm",	T_XBM },
	{ "xwd",	T_XWD },
	{ NULL,		T_FILE }		/* Default value */
};

/* Forwards */
void checkmode(const char *, mode_t, int);
int countfiles(const char *dname, struct cf *);
int createthumb(struct webfile *, const char *, struct cf *);
char *findsection(struct webfile *, struct cf *);
void header(FILE *, struct cf *);
void hookupthumbnails(struct cf *, struct webfile *, int);
int invisible(const char *, struct cf *, int);
int isthumbfn(const char *);
int main(int, char **);
void parsethumbnail(struct cf *, struct webfile *);
size_t process(const char *dname);
void processfile(FILE *, struct webfile *, struct cf *,
    const char *, const char *, const char *);
int reapthumbnails(const char *, struct cf *, struct webfile *, int);
int shellcmd(const char *, int *);
void shellsig(int);
void sortwebfiles(struct cf *, struct webfile *, int);
const char *thumbfn(struct webfile *, struct cf *);
const char *thumbfn2fn(const char *, struct cf *);
void usage(void) __attribute__((noreturn));
int websort_lex(const void *, const void *);
int websort_revlex(const void *, const void *);
int websort_time(const void *, const void *);
int websort_revtime(const void *, const void *);

int
main(int argc, char **argv)
{
	char *cp;
	const char *p;
	int i, d, op;
	mode_t nmode, omode;

	if ((cp = strrchr(argv[0], '/')) != NULL)
		prog = cp + 1;
	else
		prog = argv[0];

	while ((op = getopt(argc, argv, "dfrvc:")) != EOF)
		switch (op) {

		case 'c':
			configfn = optarg;
			break;

		case 'd':
			++debug;
			break;

		case 'f':
			++fflag;
			break;

		case 'r':
			++recurse;
			break;

		case 'v':
			++verbose;
			break;

		default:
			usage();
		}

	/* Turn on read and execute for group and other */
	omode = umask(022);
	nmode = omode & ~055;
	if (nmode == omode)
		(void)umask(omode);
	else {
		if (verbose)
			fprintf(stderr, "%s: changing umask from 0%o to 0%o\n",
			    prog, (int)omode, (int)nmode);
		(void)umask(nmode);
	}

	if (argc == optind) {
		affix[0] = '\0';
		(void)process(".");
	} else {
		for (i = optind; i < argc; ++i) {
			if (isdot(argv[i]))
				affix[0] = '\0';
			else {
				/* Insure trailing slash */
				strlcpy(affix, argv[i], sizeof(affix));
				p = affix + strlen(affix) - 1;
				if (p >= affix && *p != '/')
					strlcat(affix, "/", sizeof(affix));
			}
			if (pushd(argv[i], NULL, &d)) {
				(void)process(".");
				(void)popd(NULL, d);
			}
		}
	}

	exit(errors != 0);
}

/* Process a directory */
size_t
process(const char *dname)
{
	const char *p, *indexfn;
	const char *table, *tr, *tail;
	char *cp, *section, *prefixend;
	int i, row, col, force, intable;
	struct timespec file_tim, ind_tim;
	struct webfile *wp, *webfiles;
	struct cf *cf;
	struct section *sp;
	size_t cc, nfiles, nwebfiles;
	struct timespec mtim, ctim;
	char *otz;
	int d, d2;
	DIR *dirp;
	FILE *f;
	DIRENTRY *dp;
	struct stat sbuf;
	char buf[128];

	if (verbose > 1)
		fprintf(stderr, "%s: processing %s%s%s\n",
		    prog, affix, prefix, dname);

	/* Initialization */
	dirp = NULL;
	f = NULL;
	cf = NULL;
	reason = "";
	nfiles = 0;
	d = -1;
	webfiles = NULL;
	file_tim.tv_sec = 0;
	file_tim.tv_nsec = 0;
	ind_tim.tv_sec = 0;
	ind_tim.tv_nsec = 0;
	force = 0;
	indexfn = defindexfn;
	table = "<table class=\"w\">\n";
	tr = "<tr>";
	tail = "\n";
	otz = NULL;
	currenttz = NULL;

	/* Save our current working directory */
	if (!pushd(dname, &prefixend, &d))
		goto bail;

	/* Save the timezone */
	otz = getenv(tz);
	if (otz != NULL) {
		otz = strdup(otz);
		if (otz == NULL) {
			fprintf(stderr, "%s: process: strdup: %s\n",
			    prog, strerror(errno));
			exit(1);
		}
	}

	/* Read configuration info */
	cf = parsecf(configfn);
	if (cf->c_indexfn != NULL)
		indexfn = cf->c_indexfn;

	/* Configured strings */
	if (cf->c_tr != NULL)
		tr = cf->c_tr;
	if (cf->c_tail != NULL)
		tail = cf->c_tail;

	/* Uppercase thumbnail extensions? */
	if ((cf->c_flags & C_UPCASESUFFIX) == 0)
		cf->c_tnsuffix = tnlowsuffix;
	else
		cf->c_tnsuffix = tnuppersuffix;
	cf->c_tnsuffixlen = strlen(cf->c_tnsuffix);

	/* Open the directory for scanning */
	dirp = opendir(".");
	if (dirp == NULL) {
		fprintf(stderr, "%s: process: opendir %s: %s%s\n",
		    prog, affix, prefix, strerror(errno));
		++errors;
		goto bail;
	}

	if (lstat(indexfn, &sbuf) < 0) {
		/* OK if index.html doesn't exist */
		if (errno != ENOENT) {
			fprintf(stderr, "%s: process: lstat %s%s%s: %s\n",
			    prog, affix, prefix, indexfn, strerror(errno));
			++errors;
			goto bail;
		}
		++force;
		/* don't need to list a reason */
	} else {
		/* If it doesn't look like we can update the index file, bail */
		if ((sbuf.st_mode & S_IFMT) == S_IFLNK) {
			fprintf(stderr, "%s: process: %s%s%s: is a symlink\n",
			    prog, affix, prefix, indexfn);
			++errors;
			goto bail;
		}
		if ((sbuf.st_mode & S_IFMT) != S_IFREG) {
			fprintf(stderr,
			    "%s: process: %s%s%s: not a regular file\n",
			    prog, affix, prefix, indexfn);
			++errors;
			goto bail;
		}

		/* Get index mod time */
		ind_tim = stattim(&sbuf);
		checkmode(indexfn, sbuf.st_mode, T_HTML);

		/* If index file wasn't created by us, don't update it */
		f = fopen(indexfn, "r");
		if (f == NULL) {
			fprintf(stderr, "%s: process: fopen 1 %s%s%s: %s\n",
			    prog, affix, prefix, indexfn, strerror(errno));
			++errors;
			goto bail;
		}
		cc = fread(buf, 1, sizeof(buf), f);
		if (cc == 0 && feof(f)) {
			/* Zero length; force an index update */
			++force;
			reason = " (zero length)";
		} else if (strstr(buf, magic) == NULL) {
			fprintf(stderr,
			    "%s: skipping %s%s%s (not created by %s)\n",
			    prog, affix, prefix, indexfn, prog);
			++errors;
			goto bail;
		}
		(void)fclose(f);
		f = NULL;
	}

	/* Get number of directory entries */
	nwebfiles = 0;
	while ((dp = readdir(dirp)) != NULL)
		++nwebfiles;

	/* Leave a little extra room */
	nwebfiles += 8;

	/* Allocate space */
	webfiles = (struct webfile *)calloc(nwebfiles, sizeof(*wp));
	if (webfiles == NULL) {
		fprintf(stderr, "%s: process: calloc: %s\n",
		    prog, strerror(errno));
		exit(1);
	}

	/* Build the array of files */
	wp = webfiles;
	nwebfiles = 0;
	rewinddir(dirp);
	while ((dp = readdir(dirp)) != NULL) {
		/* Skip the current directory */
		if (strcmp(dp->d_name, ".") == 0)
			continue;

		if (lstat(dp->d_name, &sbuf) < 0) {
			fprintf(stderr, "%s: process: lstat %s%s%s: %s\n",
			    prog, affix, prefix, dp->d_name, strerror(errno));
			++errors;
			/* XXX skip these */
			continue;
		}

		/* If this is a symlink, use the newest timestamp */
		/* XXX could do a loop here to deal with symlink chains */
		if (S_ISLNK(sbuf.st_mode)) {
			mtim = sbuf.st_mtim;
			ctim = sbuf.st_mtim;
			if (stat(dp->d_name, &sbuf) < 0) {
				fprintf(stderr,
				    "%s: process: stat %s%s%s: %s\n",
				    prog, affix, prefix, dp->d_name,
				    strerror(errno));
				++errors;
				/* XXX skip these */
				continue;
			}
			if (tscmp(&sbuf.st_mtim, &mtim, <))
				sbuf.st_mtim = mtim;
			if (tscmp(&sbuf.st_ctim, &ctim, <))
				sbuf.st_ctim = ctim;
		}

		/* Skip the naildir */
		if (S_ISDIR(sbuf.st_mode) &&
		    matchfileordir(dp->d_name, naildir(cf), 1))
			continue;

		strlcpy(wp->w_name, dp->d_name, sizeof(wp->w_name));
		wp->w_mode = sbuf.st_mode;
		wp->w_kbytes = C2K(sbuf.st_size);

		/* It's either a file or a directory */
		if (!S_ISDIR(sbuf.st_mode)) {
			p = strrchr(wp->w_name, '.');
			if (p != NULL)
				++p;
			else
				p = "";
			wp->w_type = str2valcase(suffix2type, p);
			wp->w_tim = stattim(&sbuf);

			/* Mark invisible files */
			if (invisible(wp->w_name, cf, 0))
				wp->w_flags |= W_INVISIBLE;
		} else {
			/* Mark invisible directories */
			if (invisible(wp->w_name, cf, 1))
				wp->w_flags |= W_INVISIBLE;

			/* Don't recurse down or count files for parent dir */
			if (matchfileordir(wp->w_name, "..", 1)) {
				wp->w_type = T_PARENT;
			} else {
				/*
				 * XXX
				 * Using countfiles() isn't perfect
				 * if the "noparent" option isn't
				 * uniform across directories.
				 */
				wp->w_type = T_DIR;
				if (!WEBFILE_ISINVISIBLE(wp)) {
					if (recurse) {
						if (pushd(wp->w_name, NULL,
						    &d2)) {
							wp->w_nfiles =
							    process(".");
							(void)popd(NULL, d2);
						}
					} else {
						wp->w_nfiles =
						    countfiles(wp->w_name, cf);
					}
				}
				wp->w_tim = stattim(&sbuf);
			}
			(void)strlcat(wp->w_name, "/", sizeof(wp->w_name));
		}
		++wp;
		++nwebfiles;
	}

	/* Transfer configured timestamps */
	if (cf->c_timestamps != NULL) {
		for (wp = webfiles, i = 0; i < nwebfiles; ++i, ++wp) {
			if ((cp = getlistval(cf->c_timestamps,
			    wp->w_name, 0)) != NULL) {
				wp->w_timestamp = (time_t)atol(cp);
			}
		}
	}

	/* Sort names; we must do this before calling hookupthumbnails() */
	sortwebfiles(cf, webfiles, nwebfiles);

	/* If we have user configured thumbnails, hook them up */
	hookupthumbnails(cf, webfiles, nwebfiles);

	for (wp = webfiles, i = 0; i < nwebfiles; ++i, ++wp) {
		/*
		 * Skip invisible files that are not referenced thumbnails
		 * Complain about visible files with bad permissions
		 */
		if (WEBFILE_ISINVISIBLE(wp)) {
			if (!WEBFILE_ISREFERENCED(wp))
				continue;
		} else
			checkmode(wp->w_name, wp->w_mode, wp->w_type);

		/* Keep track of oldest file */
		if (tscmp(&file_tim, &wp->w_tim, <))
			file_tim = wp->w_tim;

		/* Collect file width and height and do icon bookkeeping */
		parsethumbnail(cf, wp);
	}

	/* Reap thumbnails (and collect thumbnail dimensions) */
	if (reapthumbnails(naildir(cf), cf, webfiles, nwebfiles))
		++force;

	/* Now check for missing thumbnails */
	for (wp = webfiles, i = 0; i < nwebfiles; ++i, ++wp) {
		if (!WEBFILE_ISINVISIBLE(wp))
			++nfiles;
		if ((wp->w_flags & W_HAVETN) != 0 &&
		    (wp->w_tnw == 0 || wp->w_tnh == 0)) {
			++force;
			reason = " (thumbnails foo)";
			break;
		}
	}

	/*
	 * Don't update the index file if we don't need to.
	 * Reason we update are:
	 *
	 *    - something got updated or reaped (force set)
	 *    - user specified the -f flag
	 *    - index file missing or zero length (no ind_time)
	 *    - config file newer than index file
	 *    - newest file in directory newer than index file
	 *
	 */
	if (!(force || fflag || ind_tim.tv_sec == 0 ||
	    tscmp(&cf->c_tim, &ind_tim, >) || tscmp(&file_tim, &ind_tim, >))) {
		goto bail;
	}

	if (verbose)
		fprintf(stderr, "%s: %s %s%s%s%s\n",
		    prog, ind_tim.tv_sec == 0 ? "creating" : "updating",
		    affix, prefix, indexfn, reason);

	/* Create directory if it doesn't exist */
	if (access(naildir(cf), F_OK) < 0) {
		if (verbose)
			fprintf(stderr, "%s: creating %s%s%s/\n",
			    prog, affix, prefix, naildir(cf));
		if (mkdir(naildir(cf), 0777) < 0) {
			fprintf(stderr, "%s: process: mkdir %s: %s\n",
			    prog, naildir(cf), strerror(errno));
			exit(1);
		}
	}

	f = fopen(indexfn, "w");
	if (f == NULL) {
		fprintf(stderr, "%s: process: fopen 2 %s%s%s: %s\n",
		    prog, affix, prefix, indexfn, strerror(errno));
		++errors;
		goto bail;
	}

	/* Start spewing html */
	header(f, cf);
	col = 0;
	row = 0;
	intable = 0;
	for (wp = webfiles, i = 0; i < nwebfiles; ++i, ++wp) {
		if (WEBFILE_ISINVISIBLE(wp))
			continue;

		/* New row logic */
		section = findsection(wp, cf);
		if (intable && (col >= cf->c_maxcol || section != NULL)) {
			/* Output last cell (which may span columns) */
			if (col < cf->c_maxcol && row > 0) {
				if (cf->c_maxcol - col > 1)
					fprintf(f, "<td colspan=%d><br></td>",
					    cf->c_maxcol - col);
				else
					fprintf(f, "<td><br></td>");
			}
			fprintf(f, "</tr>\n");

			/* Avoid huge tables (which take forever to render) */
			if (++row > cf->c_maxrow || section != NULL) {
				fprintf(f, "</table>\n");
				row = 0;
				intable = 0;
			} else
				fprintf(f, "%s", tr);
			col = 0;
		}
		if (section != NULL) {
			fprintf(f, "%s", section);
			free(section);
		}
		if (!intable) {
			fprintf(f, "%s%s",
			    cf->c_table != NULL ? cf->c_table : table, tr);
			++intable;
		}
		fprintf(f, "<td>");
		++col;
		processfile(f, wp, cf, table, tr, indexfn);
		fprintf(f, "</td>\n");
	}
	if (intable) {
		/* Output last cell (which may span columns) */
		if (col < cf->c_maxcol && row > 0) {
			if (cf->c_maxcol - col > 1)
				fprintf(f, "<td colspan=%d><br></td>",
				    cf->c_maxcol - col);
			else
				fprintf(f, "<td><br></td>");
		}
		fprintf(f, "</tr></table>");
		intable = 0;
	}
	fprintf(f, "%s</body></html>\n", tail);

	/* Look for orphan sections */
	for (sp = cf->c_section; sp != NULL; sp = sp->s_next)
		fprintf(stderr, "%s: warning: %s%s%s: section not found\n",
		    prog, affix, prefix, sp->s_file);

	/* Look for orphan list elements */
	reportunused(cf->c_alttext, "unused alttext");
	reportunused(cf->c_captions, "unused caption");
	/* XXX should we check cf->c_invisible? */
	reportunused(cf->c_thumbnails, "unused thumbnail");
	reportunused(cf->c_timestamps, "unused timestamp");

bail:
	/* Close the directory */
	if (dirp != NULL && closedir(dirp) < 0) {
		fprintf(stderr, "%s: process: closedir: %s\n",
		    prog, strerror(errno));
		++errors;
	}

	/* Restore our current working directory */
	(void)popd(prefixend, d);

	/* Restore timezone */
	if (otz != NULL) {
		setenv(tz, otz, 1);
		free(otz);
	} else {
#ifdef HAVE_UNSETENV
		unsetenv(tz);
#else
		setenv(tz, "", 1);
#endif
	}

	/* Close the index file */
	if (f != NULL && fclose(f) == EOF) {
		fprintf(stderr, "%s: process: fclose %s%s%s: %s\n",
		    prog, affix, prefix, indexfn, strerror(errno));
		++errors;
	}

	/* Free the file entries */
	if (webfiles != NULL)
		free(webfiles);

	/* Free the config */
	if (cf != NULL)
		freecf(cf);

	return (nfiles);
}

/* Process a file */
void
processfile(FILE *f, struct webfile *wp, struct cf *cf,
    const char *table, const char *tr, const char *indexfn)
{
	char *cp, *cp2, *name;
	const char *p, *icon, *alt, *linkpostfix, *newtz;
	struct webfile *wp2;
	struct list *lp;
	int i, j, k, w, h;
	time_t t;
	struct tm *tm;
	char buf[132];
	char target[256];
	char location[32];
	char url[64];

	if (WEBFILE_ISDIR(wp) && (cf->c_flags & C_APPDIRINDEXFN) != 0)
		linkpostfix = indexfn;
	else
		linkpostfix = "";

	target[0] = '\0';
	if (cf->c_target != NULL)
		(void)snprintf(target, sizeof(target), " target=\"%s\"",
		    cf->c_target);

	/* Start the table and the image link */
	fprintf(f, "%s%s<td><a href=\"%s%s\"%s>",
	    table, tr, wp->w_name, linkpostfix, target);

	/* Get the icon or thumbnail */
	if (wp->w_tn != NULL)
		wp2 = wp->w_tn;			/* use configured thumbnail */
	else
		wp2 = wp;
	switch (wp2->w_type) {

	case T_AVI:
	case T_DIR:
	case T_FILE:
	case T_FLASH:
	case T_HTML:
	case T_MP3:
	case T_PARENT:
	case T_PDF:
	case T_QT:
	case T_REAL:
		/* XXX This is kind of weird if we're using a configured tn */
		icon = createicon(wp2->w_type, cf, &w, &h);
		break;

	case T_BMP:
	case T_GIF:
	case T_JPEG:
	case T_PBM:
	case T_PNG:
	case T_TIFF:
	case T_XBM:
	case T_XWD:
		icon = thumbfn(wp2, cf);
		if (wp2->w_tnw > 0 && wp2->w_tnh > 0) {
			w = wp2->w_tnw;
			h = wp2->w_tnh;
		} else {
			/* Create/update the thumbnail image */
			if (verbose)
				fprintf(stderr, "%s: %s %s%s%s\n",
				    prog, wp2->w_tn_tim.tv_sec == 0 ?
				    "creating" : "updating",
				    affix, prefix, icon);
			if (createthumb(wp2, icon, cf)) {
				w = wp2->w_tnw;
				h = wp2->w_tnh;
			} else {
				w = 1;
				h = 1;
			}
		}
		break;

	default:
		fprintf(stderr, "%s: processfile: unknown type %d\n",
		    prog, wp2->w_type);
		icon = "?";
		h = w = cf->c_iconsize;
		++errors;
		break;
	}

	/* Figure out alt and and file label names */
	name = wp->w_name;
	if (wp->w_type == T_PARENT)
		name = "Parent dir";
	if ((cp = getlistval(cf->c_alttext, wp->w_name, 1)) != NULL)
		alt = cp;
	else
		alt = name;

	fprintf(f, "<img width=%d height=%d", w, h);
	/* Only need inline css if we have a border or a small thumbnail */
	i = (cf->c_iconsize - h) / 2;
	if (cf->c_imgborder > 0 || i > 0) {
		fprintf(f, " style=\"");
		if (cf->c_imgborder > 0) {
			fprintf(f, "border: %dpx;", cf->c_imgborder);
			if (i > 0)
				fputc(' ', f);
		}
		/* Make text line up when the thumbnail is small */
		if (i > 0)
			fprintf(f, "margin: %dpx 0;", i);
		fputc('"', f);
	}

	fprintf(f, "\n    src=\"%s\" alt=\"%s\"></a></td></tr>\n", icon, alt);

	/* Filename (break long ones) */
	if (cf->c_namelen >= 0 && (cf->c_flags & C_NO_NAME) == 0) {
		fprintf(f, "%s<td><a href=\"%s%s\"%s><small>",
		    tr, wp->w_name, linkpostfix, target);
		cp = name;
		/* Break long names */
		if (cf->c_namelen != 0) {
			i = strlen(cp);
			while (i > cf->c_namelen) {
				j = cf->c_namelen;
				/* Look for a natural break */
				for (k = j; k > cf->c_namelen / 2; --k)
					if (!isalnum(cp[k - 1])) {
						j = k;
						break;
					}
				/*
				 * Avoid non alpha/num at line
				 * start and single alpha/num line.
				 */
				if (!isalnum(cp[j]))
					--j;
				else if (i - j == 1 && isalnum(cp[j - 1]))
					--j;
				/* Always output at least one character! */
				if (j <= 0)
					j = 1;
				fprintf(f, "%.*s<br>", j, cp);
				i -= j;
				cp += j;
			}
		}
		fprintf(f, "%s</small></a></td></tr>\n", cp);
	}

	if (wp->w_type == T_DIR) {
		if ((cf->c_flags & C_NO_DIRDETAIL) == 0)
			fprintf(f, "%s<td><small>%d file%s</small></td></tr>\n",
			    tr, wp->w_nfiles, PLURAL(wp->w_nfiles));
	}
	else if (wp->w_type != T_PARENT) {
		/* image w/h */
		if ((wp->w_w > 0 || wp->w_h > 0) &&
		    (cf->c_flags & C_NO_IMGSIZE) == 0)
			fprintf(f, "%s<td><small>%dx%d</small></td></tr>\n",
			    tr, wp->w_w, wp->w_h);

		/* file size */
		cp = buf;
		*cp = '\0';
		if ((cf->c_flags & C_NO_SIZE) == 0) {
			if (wp->w_kbytes < 10 * 1024)
				sprintf(cp, "%dK", wp->w_kbytes);
			else
				sprintf(cp, "%dM", C2K(wp->w_kbytes));
			cp += strlen(cp);
			*cp = '\0';
		}

		if (wp->w_timestamp > 0) {
			newtz = gettz(cf, wp->w_timestamp);
			if (currenttz != newtz) {
				currenttz = newtz;
				setenv(tz, currenttz, 1);
			}

			/* XXX timezone correction hack */
			t = wp->w_timestamp;
			if (cf->c_tzhours != 0)
				t += cf->c_tzhours * 60 * 60;
			tm = localtime(&t);
			if (cf->c_datefmt != NULL)
				p = cf->c_datefmt;
			else
				p = defdatefmt;
			if (cp > buf)
				*cp++ = ' ';
			*cp = '\0';
			(void)strftime(cp, sizeof(buf) - strlen(buf), p, tm);
		} else if (wp->w_timestamp == 0 && cf->c_timestamps != NULL)
			fprintf(stderr,
			    "%s: warning: %s%s%s: missing timestamp\n",
			    prog, affix, prefix, wp->w_name);
		cp = buf;
		if (*cp != '\0')
			fprintf(f, "%s<td><small>%s</small></td></tr>\n",
			    tr, cp);

		/* location */
		if (wp->w_location[0] != '\0') {
			/* url location */
			strlcpy(location, wp->w_location, sizeof(location));
			cp2 = strchr(location, ' ');
			if (cp2 != NULL) {
				i = wp->w_location - cp2;
				j = strlen(cp + 1) + 1;
				memmove(cp2 + 3, cp2 + 1, j);
				cp2[0] = '%';
				cp2[1] = '2';
				cp2[2] = '0';
			}
			(void)snprintf(url, sizeof(url), gmapsfmt, location);
			fprintf(f, "%s<td><a href=\"%s\" "
			    "class=\"location\"></a></td></tr>\n", tr, url);
		}

	}

	/* Look for a caption */
	if (cf->c_captions != NULL)
		for (lp = cf->c_captions; lp->l_key != NULL; ++lp) {
			if (matchfileordir(wp->w_name, lp->l_key,
			    WEBFILE_ISDIR(wp))) {
				cp = lp->l_val;
				if (*lp->l_val != '\0')
					fprintf(f, "%s<td><small>%s</small>"
					    "</td></tr>\n", tr, lp->l_val);
				lp->l_flags |= L_REFERENCED;
				break;
			}
		}
	fprintf(f, "</table>");
}

void
sortwebfiles(struct cf *cf, struct webfile *wp, int n)
{
	int (*websort)(const void *, const void *);
	int needtimestamps;

	needtimestamps = 0;
	switch (cf->c_sort) {

	case C_SORT_LEX:
		websort = websort_lex;
		break;

	case C_SORT_REVLEX:
		websort = websort_revlex;
		break;

	case C_SORT_TIME:
		websort = websort_time;
		++needtimestamps;
		break;

	case C_SORT_REVTIME:
		websort = websort_revtime;
		++needtimestamps;
		break;

	default:
		fprintf(stderr, "%s: unknown sort type %d\n", prog, cf->c_sort);
		exit(1);
	}
	if (needtimestamps && cf->c_timestamps == NULL)
		fprintf(stderr, "%s: warning: "
		    "Sorting on timestamps but none configured\n", prog);
	qsort(wp, n, sizeof(*wp), websort);
}

void
hookupthumbnails(struct cf *cf, struct webfile *webfiles, int nfiles)
{
	int i, j;
	char *name;
	struct webfile *wp, *wp2;
	struct list *lp;

	if (cf->c_thumbnails == NULL)
		return;

	for (wp = webfiles, i = 0; i < nfiles; ++i, ++wp) {
		name = NULL;
		for (lp = cf->c_thumbnails; lp->l_key != NULL; ++lp)
			if (matchfileordir(wp->w_name, lp->l_key,
			    WEBFILE_ISDIR(wp))) {
				name = lp->l_val;
				lp->l_flags |= L_REFERENCED;
				break;
			}
		if (name == NULL)
			continue;
		for (wp2 = webfiles, j = 0; j < nfiles; ++j, ++wp2) {
			if (matchfileordir(wp2->w_name, name,
			    WEBFILE_ISDIR(wp2))) {
				wp->w_tn = wp2;
				wp2->w_flags |= W_REFERENCED;
				break;
			}
		}
		if (wp->w_tn == NULL)
			fprintf(stderr, "%s: warning:"
			    " Couldn't find configured thumbnail: %s\n",
				prog, name);
	}
}

void
parsethumbnail(struct cf *cf, struct webfile *wp)
{
	int v, flags;
	char *location;
	size_t locationsize;

	flags = 0;
	switch (wp->w_type) {

	case T_BMP:
		if (parsebmp(wp->w_name, &wp->w_w, &wp->w_h, NULL))
			flags |= W_HAVETN;
		else
			wp->w_type = T_FILE;
		break;

	case T_GIF:
		if (parsegif(wp->w_name, &wp->w_w, &wp->w_h, NULL))
			flags |= W_HAVETN;
		else
			wp->w_type = T_FILE;
		break;

	case T_JPEG:
		if ((cf->c_flags & C_NO_LOCATION) == 0) {
			location = wp->w_location;
			locationsize = sizeof(wp->w_location);
		} else {
			location = NULL;
			locationsize = 0;
		}
		if (parsejpeg(wp->w_name, &wp->w_w, &wp->w_h, NULL,
		    location, locationsize))
			flags |= W_HAVETN;
		else
			wp->w_type = T_FILE;
		break;

	case T_PBM:
#ifdef HAVE_LIBPPM
		if (parsepbm(wp->w_name, &wp->w_w, &wp->w_h, NULL))
			flags |= W_HAVETN;
		else
#endif
			wp->w_type = T_FILE;
		break;

	case T_PNG:
		if (parsepng(wp->w_name, &wp->w_w, &wp->w_h, NULL))
			flags |= W_HAVETN;
		else
			wp->w_type = T_FILE;
		break;

	case T_TIFF:
#ifdef HAVE_LIBTIFF
		if (parsetiff(wp->w_name, &wp->w_w, &wp->w_h, NULL))
			flags |= W_HAVETN;
		else
#endif
			wp->w_type = T_FILE;
		break;

	case T_XBM:
		if (parsexbm(wp->w_name, &wp->w_w, &wp->w_h, NULL))
			flags |= W_HAVETN;
		else
			wp->w_type = T_FILE;
		break;

	case T_XWD:
		if (parsexwd(wp->w_name, &wp->w_w, &wp->w_h, NULL))
			flags |= W_HAVETN;
		else
			wp->w_type = T_FILE;
		break;

	default:
		/* Turn on icon bit */
		v = typ2flag(wtyp2bit, wp->w_type);
		if (wp->w_tn == NULL &&
		    (v != C_FILE_ICON || !WEBFILE_ISINVISIBLE(wp)))
			cf->c_icons |= v;
		break;
	}
	if (!WEBFILE_ISINVISIBLE(wp) && WEBFILE_ISREFERENCED(wp))
		wp->w_flags |= flags;
}

/* As a side effect, we collect the thumbnail sizes */
int
reapthumbnails(const char *dir, struct cf *cf,
    struct webfile *webfiles, int nfiles)
{
	int i, v, force, ignorefile, icons;
	struct webfile *wp;
	const char *p, *file;
	int w, h;
	DIR *dirp;
	DIRENTRY *dp;
	char temp[1024];

	errno = 0;
	if ((dirp = opendir(dir)) == NULL) {
		if (errno != ENOENT) {
			fprintf(stderr,
			    "%s: reapthumbnails: opendir %s%s%s: %s\n",
			    prog, affix, prefix, dir, strerror(errno));
			++errors;
		}
		return (1);
	}

	force = 0;
	icons = 0;
	while ((dp = readdir(dirp)) != NULL) {
		if (strcmp(dp->d_name, ".") == 0 ||
		    strcmp(dp->d_name, "..") == 0)
			continue;
		ignorefile = 0;
		if (*dp->d_name == '.') {
			/* Icon files start with '.'; get icon flag value */
			v = str2val(str2icon, dp->d_name);
			/* Ignore non-icons, skip to next file if needed icon */
			if (v == 0) {
				p = strrchr(dp->d_name, '.');
				/* ignore if not a gif file */
				if (p == NULL || strcmp(p + 1, "gif") != 0)
					++ignorefile;
			} else {
				icons |= v;
				/* Reap if not one we need */
				if ((v & cf->c_icons) != 0)
					goto nextfile;
			}
			/* Fall through and reap (or ignore) file... */
		} else if ((file = thumbfn2fn(dp->d_name, cf)) == NULL) {
			/*
			 * Reap things that look like icons
			 * Handle both upper and lowercase extensions
			 * Only ignore if it doesn't have a thumbnail suffix
			 */
			if (!isthumbfn(dp->d_name))
				++ignorefile;
			/* Fall through and reap (or ignore) file... */
		} else for (wp = webfiles, i = 0; i < nfiles; ++i, ++wp) {
			if (strcmp(file, wp->w_name) != 0)
				continue;
			if (WEBFILE_ISINVISIBLE(wp) &&
			    !WEBFILE_ISREFERENCED(wp))
				break;

			/* Keep track of icons */
			icons |= typ2flag(wtyp2bit, wp->w_type);

			/* Get thumbnail dimensions and timestamp */
			sprintf(temp, "%s/%s", dir, dp->d_name);
			if (!parsejpeg(temp, &w, &h, &wp->w_tn_tim, NULL, 0)) {
				++force;
				reason = " (parsejpeg)";
			} else if (wp->w_tn_tim.tv_sec == 0) {
				++force;
				reason = " (no thumbnail time)";
			} else if (wp->w_tn == NULL &&
			    tscmp(&wp->w_tn_tim, &wp->w_tim, < )) {
				/* Skip check when a configured thumbnail */
				++force;
				reason = " (thumbnail out of date)";
			} else if (w > cf->c_iconsize || h > cf->c_iconsize ||
			    !(w == cf->c_iconsize || h == cf->c_iconsize)) {
				++force;
				reason = " (thumbnail dimensions)";
			} else {
				/* Salt away icon dimensions */
				wp->w_tnw = w;
				wp->w_tnh = h;
			}
			goto nextfile;
		}

		/* Ignore file (i.e. gripe but don't delete) */
		if (ignorefile) {
			fprintf(stderr,
			    "%s: ignoring %s%s%s/%s (unknown file)\n",
			    prog, affix, prefix, dir, dp->d_name);
			++force;
			reason = " (ignore file)";
			continue;
		}

		/* Reap file */
		sprintf(temp, "%s/%s", dir, dp->d_name);
		if (verbose)
			fprintf(stderr, "%s: deleting %s%s%s\n",
			    prog, affix, prefix, temp);
		if (unlink(temp) < 0)
			fprintf(stderr,
			    "%s: reapthumbnails: unlink %s%s%s: %s\n",
			    prog, affix, prefix, temp, strerror(errno));
		++force;
		reason = " (reap file)";
nextfile:
		continue;
	}

	(void)closedir(dirp);

	/* Check if we're missing any icons */
	if ((~icons & cf->c_icons) != 0) {
		++force;
		reason = " (missing icons)";
	}
	return (force);
}

void
header(FILE *f, struct cf *cf)
{
	const char *p;
	struct tm *tm;
	time_t now;
	char date[64];
	char zone[32];
#ifdef HAVE_ALTZONE
	int z;
#endif

	now = time(NULL);
	tm = localtime(&now);
	(void)strftime(date, sizeof(date), "%d-%b-%Y %T", tm);
#ifndef HAVE_ALTZONE
	(void)strftime(zone, sizeof(zone), "%z", tm);
#else
	z = altzone / -36;
	if (z < 0) {
		z = -z;
		zone[0] = '-';
	} else
		zone[0] = '+';
	(void)sprintf(zone + 1, "%04d", z);
#endif


	fprintf(f, "<!DOCTYPE html>\n");
	fprintf(f, "<!-- %s\n    Version %s (this file created on %s %s) -->\n",
	    magic, version, date, zone);
	fprintf(f, "<html lang=\"en\"><head>\n");
	fprintf(f, stylefmt, debug ? "    border: 1px solid;\n" : "");
	if (cf->c_favicon != NULL)
		fprintf(f, "<link rel=\"shortcut icon\" href=\"%s\">\n",
		    cf->c_favicon);
	if (cf->c_css != NULL)
		fprintf(f, "<link rel=\"stylesheet\" type=\"text/css\""
		    " href=\"%s\">\n", cf->c_css);
	fprintf(f, "<meta http-equiv=\"Content-type\""
	    " content=\"text/html;charset=UTF-8\">\n");
	fprintf(f, "<meta name=\"viewport\""
	    " content=\"width=device-width, initial-scale=1\">\n");
	fprintf(f, "<title>\n");
	if (cf->c_title != NULL)
		fprintf(f, "%s", cf->c_title);
	else
		fprintf(f, "%s\n", simplewd());
	fprintf(f, "</title></head>\n");
	if (cf->c_body != NULL)
		fprintf(f, "%s", cf->c_body);
	else
		fprintf(f, "<body>\n");

	/* If the user supplied a header use it. Otherwise use the title */
	if (cf->c_head != NULL)
		fprintf(f, "%s", cf->c_head);
	else {
		if (cf->c_title != NULL)
			p = cf->c_title;
		else
			p = simplewd();
		fprintf(f, "<h2>%s</h2>", p);
	}
}

int
createthumb(struct webfile *wp, const char *fn, struct cf *cf)
{
	const char *what;
	char cmd[2048];
	int w, h, status;

	switch (wp->w_type) {

	case T_BMP:
		what = "bmptoppm";
		break;

	case T_GIF:
		what = "giftopnm";
		break;

	case T_JPEG:
		what = "djpeg";
		break;

	case T_PBM:
		what = NULL;
		break;

	case T_PNG:
		what = "pngtopnm";
		break;

	case T_TIFF:
		what = "tifftopnm";
		break;

	case T_XWD:
		what = "xwdtopnm";
		break;

	case T_XBM:
		what = "xbmtopbm";
		break;

	default:
		fprintf(stderr, "%s: createthumb: unknown type %d\n",
		    prog, wp->w_type);
		++errors;
		return (0);
	}
	if (what == NULL)
		sprintf(cmd, "pnmscale -xysize %d %d %s|cjpeg >",
		    cf->c_iconsize, cf->c_iconsize, escapename(wp->w_name));
	else
		sprintf(cmd, "%s %s|pnmscale -xysize %d %d|cjpeg >", what,
		    escapename(wp->w_name), cf->c_iconsize, cf->c_iconsize);
	strlcat(cmd, escapename(fn), sizeof(cmd));

	if (verbose > 1)
		fprintf(stderr, "%s: \"%s\"\n", prog, cmd);
	status = 0;
	errno = 0;
	if (shellcmd(cmd, &status) < 0 || status != 0) {
		if (killed) {
			fprintf(stderr, "\n");
			exit(1);
		}
		if (errno != 0)
			fprintf(stderr,
			    "%s: createthumb: \"%s ...\" failed: %s\n",
			    prog, what, strerror(errno));
		fprintf(stderr, "%s: deleting %s%s%s", prog, affix, prefix, fn);
		if (remove(fn) < 0 && errno != ENOENT)
			fprintf(stderr, " failed: %s", strerror(errno));
		fprintf(stderr, "\n");
		return (0);
	}
	if (parsejpeg(fn, &w, &h, NULL, NULL, 0)) {
		wp->w_tnw = w;
		wp->w_tnh = h;
	} else {
		fprintf(stderr,
		    "%s: deleting %s%s%s (bad format thumbnail)\n",
			prog, affix, prefix, fn);
			++errors;
		errno = 0;
		if (unlink(fn) < 0 && errno != ENOENT)
			fprintf(stderr, "%s: createthumb: unlink %s%s%s: %s\n",
			    prog, affix, prefix, fn, strerror(errno));
	}
	return (1);
}

void
shellsig(int signo)
{

	++killed;
}

/* Like system() but catch some signals */
int
shellcmd(const char *cmd, int *sp)
{
	pid_t wpid, pid;
	int xerrno;
	void (*oint)(int), (*oterm)(int);

	killed = 0;
	oint = setsignal(SIGINT, shellsig);
	oterm = setsignal(SIGTERM, shellsig);

	wpid = fork();
	if (wpid < 0) {
		xerrno = errno;
		oint = setsignal(SIGINT, shellsig);
		oterm = setsignal(SIGTERM, shellsig);
		errno = xerrno;
		return (wpid);
	}
	if (wpid == 0) {
		/* Child */
		(void)setsignal(SIGINT, oint);
		(void)setsignal(SIGTERM, oterm);
		execl(_PATH_BSHELL, "sh", "-c", cmd, (char *)NULL);
		exit(1);
	}

	/* Parent */
	do {
		pid = waitpid(wpid, sp, 0);
	} while (pid -1 && errno == EINTR);

	xerrno = errno;
	(void)setsignal(SIGINT, oint);
	(void)setsignal(SIGTERM, oterm);
	errno = xerrno;
	if (pid < 0)
		return (pid);

	if (killed) {
		errno = 0;
		return (-1);
	}

	return (0);
}

int
invisible(const char *fn, struct cf *cf, int isdir)
{
	int len;
	const char *p, *indexfn;
	struct list *lp;

	/* Show parent directory */
	if (isdir && matchfileordir(fn, "..", 1))
		return ((cf->c_flags & C_NO_PARENT) != 0);

	/* Ignore all other dot files */
	if (*fn == '.')
		return (1);

	/* Ignore the index and config files */
	if (cf->c_indexfn != NULL)
		indexfn = cf->c_indexfn;
	else
		indexfn = defindexfn;
	if (strcmp(fn, indexfn) == 0 || strcmp(fn, configfn) == 0)
		return (1);

	/* Ignore the nail subdirectory */
	if (isdir) {
		p = naildir(cf);
		/* Don't need to check if it's a dot file */
		if (*p != '.' && matchfileordir(fn, p, 1))
			return (1);
	}

	/* Ignore files with a trailing dash */
	len = strlen(fn);
	p = fn + len - 1;
	if (p >= fn && *p == '-')
		return (1);
	if (p >= fn + 1 && p[-1] == '-' && p[0] == '/')
		return (1);

	/* Check for configured invisible files */
	if (cf->c_invisible != NULL)
		for (lp = cf->c_invisible; lp->l_key != NULL; ++lp)
			if (matchfileordir(lp->l_key, fn, isdir))
				return (1);

	/* Check for files that match invisiblepat */
	if (cf->c_invisiblepat != NULL && match(cf->c_invisiblepat, fn))
		return (1);

	return (0);
}

void
checkmode(const char *fn, mode_t mode, int wtype)
{
	mode_t smode;			/* must be set bits */
	mode_t cmode;			/* must be clear bits */

	cmode = 0;
	if (S_ISDIR(mode))
		smode = 0555;
	else {
		smode = 0444;
		if (wtype != T_FILE)
			cmode = 0111;
	}
	if ((mode & smode) != smode || (mode & cmode) != 0)
		fprintf(stderr, "%s: warning: %s%s%s: bad permission (0%o)\n",
		    prog, affix, prefix, fn, mode & 0777);
}

int
isthumbfn(const char *fn)
{
	const char *p, *ep;

	ep = fn + strlen(fn);

	p = ep - tnlowsuffixlen;
	if (p >= fn && strcmp(p, tnlowsuffix) == 0)
		return (1);

	p = ep - tnuppersuffixlen;
	if (p >= fn && strcmp(p, tnuppersuffix) == 0)
		return (1);

	return (0);
}

/* Create a thumbnail filename */
/* XXX doesn't work well when there's no extension */
const char *
thumbfn(struct webfile *wp, struct cf *cf)
{
	char *cp, *cp2;
	static char file[1024];

	sprintf(file, "%s/%s", naildir(cf), wp->w_name);
	cp2 = file + strlen(naildir(cf)) + 1;
	cp = strrchr(cp2, '.');
	if (cp == NULL) {
		/* XXX */
		cp = cp2 + strlen(cp2);
		strcpy(cp, ".xxx");
	}
	*cp++ = '_';
	if (strcmp(cp, "jpg") == 0)
		*cp = '\0';
	strlcat(file, cf->c_tnsuffix, sizeof(file));
	return (file);
}

/* Convert a thumbnail name back to the original filename */
const char *
thumbfn2fn(const char *tn, struct cf *cf)
{
	char *cp;
	static char file[1024];

	(void)strcpy(file, tn);
	cp = file + strlen(file) - cf->c_tnsuffixlen;

	/* Bail if this isn't a thumbnail */
	if (cp >= file && strcmp(cp, cf->c_tnsuffix) != 0)
		return (NULL);

	*cp-- = '\0';
	if (cp < file)
		return (NULL);
	if (*cp == '_') {
		/* "<name>__t.jpg" -> "<name>.jpg" */
		strcpy(cp, ".jpg");
	} else {
		/* "<name>_<ext>_t.jpg" -> "<name>.<ext>" */
		cp = strrchr(file, '_');
		if (cp == NULL)
			return (NULL);
		*cp = '.';
	}
	return (file);
}

int
countfiles(const char *dname, struct cf *cf)
{
	int n, isdir;
	DIR *dirp;
	DIRENTRY *dp;
	struct stat sbuf;

	n = 0;
	if ((dirp = opendir(dname)) != NULL) {
		while ((dp = readdir(dirp)) != NULL) {
			isdir = 0;
			if (stat(dp->d_name, &sbuf) >= 0 &&
			    S_ISDIR(sbuf.st_mode))
				++isdir;
			if (!invisible(dp->d_name, cf, isdir))
				++n;
		}
		(void)closedir(dirp);
	}
	return (n);
}

int
websort_lex(const void *a1, const void *a2)
{
	const struct webfile *w1, *w2;

	w1 = a1;
	w2 = a2;
	return (strcoll(w1->w_name, w2->w_name));
}

int
websort_revlex(const void *a1, const void *a2)
{
	const struct webfile *w1, *w2;

	w1 = a1;
	w2 = a2;
	return (-strcoll(w1->w_name, w2->w_name));
}

int
websort_time(const void *a1, const void *a2)
{
	const struct webfile *w1, *w2;

	w1 = a1;
	w2 = a2;
	if (w1->w_timestamp == w2->w_timestamp)
		return (websort_lex(a2, a1));
	return (w1->w_timestamp - w2->w_timestamp);
}

int
websort_revtime(const void *a1, const void *a2)
{
	const struct webfile *w1, *w2;

	w1 = a1;
	w2 = a2;
	if (w1->w_timestamp == w2->w_timestamp)
		return (websort_lex(a1, a2));
	return (w2->w_timestamp - w1->w_timestamp);
}

const char *
naildir(struct cf *cf)
{
	if (cf->c_naildir != NULL)
		return (cf->c_naildir);
	return (".nai");
}

/*
 * Find the section that goes with this filename.
 * The section data is removed from the list and freed.
 * The caller must free the returned string.
 */
char *
findsection(struct webfile *wp, struct cf *cf)
{
	struct section *sp, *sp2;
	char *cp;

	sp2 = NULL;
	for (sp = cf->c_section; sp != NULL; sp = sp->s_next) {
		if (matchfileordir(wp->w_name, sp->s_file, WEBFILE_ISDIR(wp))) {
			cp = sp->s_html;
			free(sp->s_file);
			if (sp2 != NULL)
				sp2->s_next = sp->s_next;
			else
				cf->c_section = sp->s_next;
			free(sp);
			return (cp);
		}
		sp2 = sp;
	}

	return (NULL);
}

void
usage(void)
{
	fprintf(stderr, "%s version %s\n", prog, version);
	fprintf(stderr, "usage: %s [-dfrv] [-c conf] [directory ...]\n", prog);
	if (verbose)
		fprintf(stderr, "%s", copyright);
	exit(1);
}
