#!/bin/sh
# @(#) $Id: jpegrotate.sh 369 2025-11-07 23:23:49Z leres $ (XSE)
#
# jpegrotate - rotate jpeg images
#

prog="`basename $0`"

usage()
{
	echo "${prog} file ...	# auto-rotate" 1>&2
	echo "${prog} -l file ...	# rotate left" 1>&2
	echo "${prog} -r file ...	# rotate right" 1>&2
	echo "${prog} -f file ...	# rotate 180 (flip)" 1>&2
	echo "${prog} -h file ...	# horizontal" 1>&2
	echo "${prog} -V file ...	# vertical" 1>&2
	echo "${prog} -u file ...	# undo" 1>&2
	echo "${prog} -n		# dryrun" 1>&2
	echo "${prog} -F		# force" 1>&2
	echo "${prog} -O		# override" 1>&2
	exit 1
}

PATH=/usr/local/bin:${PATH}
export PATH

automatic=1
dryrun=0
force=0
option=""
override=0
status=0
undo=0

while getopts fhlnruFOV opt ; do
	case "$opt" in

	f)
		option="-1"
		automatic=0
		;;

	h)
		option="-F"
		automatic=0
		;;

	l)
		option="-2"
		automatic=0
		;;

	n)
		dryrun=1
		;;

	r)
		option="-9"
		automatic=0
		;;

	u)
		undo=1
		;;

	F)
		force=1
		;;

	O)
		override=1
		;;

	V)
		option="-f"
		automatic=0
		;;

	?)
		usage
		;;
	esac
done

shift $((${OPTIND} - 1))

if [ -n "${option}" -a ${undo} -ne 0 ]; then
	usage
fi

if [ "${force}" -ne 0 -a ${override} -ne 0 ]; then
	echo "${prog}: Cannot use both -F (force) and -O (override)" 1>&2
	exit 1
fi

if [ -z "${option}" -a ${override} -ne 0 ]; then
	echo "${prog}: Cannot use automatic with -O (override)" 1>&2
	exit 1
fi

if [ ${automatic} -ne 0 ]; then
	option="-a"
fi

if [ $# -lt 1 ]; then
	usage
fi

# We need the sysv version under Solaris
touch="/usr/bin/touch"
if [ ! -x ${touch} ]; then
	touch="touch"
fi

# Execute (or just echo if -n)
execute() {
	cmd="$*"
	if [ ${dryrun} -eq 0 ]; then
		eval "${cmd}"
		return $?
	fi
	echo "+ ${cmd}"
	return 0
}

if [ ${undo} -ne 0 ]; then
	# Undo logic
	while [ $# -gt 0 ]; do
		r="$1"
		shift

		f="`echo ${r} |
		    sed -e 's/r\.jpg$/.jpg/' -e 's/r\.JPG$/.JPG/'`"
		if [ "${f}" = "${r}" ]; then
			echo "${prog}: ${r} already unrotated? (skipping)"
			status=1
			continue
		fi
		if [ -r ${r} -a -r ${f} ]; then
			echo "${prog}: Both ${r} and ${f} exist (skipping)"
			status=1
			continue
		fi
		if [ -r ${f} ]; then
			echo "${prog}: ${f} already exists (skipping)"
			status=1
			continue
		fi
		if [ ! -r ${r} ]; then
			echo "${prog}: Can't find ${r}"
			status=1
			continue
		fi
		o="${f}-"
		if [ ! -r ${o} ]; then
			echo "${prog}: ${o} missing (skipping)"
			status=1
			continue
		fi

		execute "rm -f ${r}-"
		execute "mv ${r} ${r}-"
		execute "mv ${o} ${f}"
		echo "${prog}: ${r} -> ${f}"
	done
	exit ${status}
fi

# Make sure we have everything we need
for f in exiftags exiftran; do
	which ${f} > /dev/null 2>&1
	if [ $? -ne 0 ]; then
		echo "${prog}: Can't find ${f}" 1>&2
		exit 1
	fi
done

# Rotate logic
while [ $# -gt 0 ]; do
	f="$1"
	shift

	case "`echo ${f} | tr A-Z a-z`" in

	*.jpg)
		# Ok
		;;

	*)
		echo "${prog}: ${f} unexpected file extension (skipping)"
		status=1
		continue
		;;
	esac

	r="`echo ${f} | sed -e 's/r*\(\.[Jj][Pp][Gg]\)$/r\1/'`"
	if [ "${f}" = "${r}" -a ${override} -eq 0 ]; then
		echo "${prog}: ${f} already rotated? (use -O to override)"
		status=1
		continue
	fi
	if [ -r ${r} -a ${override} -eq 0 ]; then
		echo "${prog}: ${r} already exists, skipping ${f}"
		status=1
		continue
	fi
	if [ ! -r ${f} ]; then
		echo "${prog}: Can't find ${f}"
		status=1
		continue
	fi
	o="${f}-"
	if [ -r ${o} -a ${force} -eq 0 ]; then
		echo "${prog}: ${o} already exists (skipping)"
		status=1
		continue
	fi

	# If we're autodetecting, see if anything needs to be done
	if [ ${automatic} -ne 0 ]; then
		rotate=`exiftags ${f} 2>&1 |
		    awk '
		    /Image Orientation: Top, Left-Hand/ {
			    print "0"
			    next
		    }
		    /Image Orientation: Right-Hand, Top/ {
			    print "90"
			    next
		    }
		    /Image Orientation: Bottom, Right-Hand/ {
			    print "180"
			    next
		    }
		    /Image Orientation: Left-Hand, Bottom/ {
			    print "270"
			    next
		    }
		    /Image Orientation: / {
			    i = index($0, ": ")
			    if (i > 0)
				    print "# " substr($0, i + 2)
			    next
		    }
		    /exiftags: maker .* not supported/ {
			    # ignore
			    next
		    }
		    /exiftags: / {
			    # Print errors
			    print
		    }
		    '`
		case "${rotate}" in

		0)
			# No transformation needed
			continue
			;;

		[0-9]*)
			# Ok!
			;;

		"")
			echo "${prog}: Can't parse ${f} (skipping)" 1>&2
			continue
			;;

		*)
			echo "${prog}: Can't parse ${f} (rotate: ${rotate}, skipping)" 1>&2
			continue
			;;
		esac
	fi

	# Pick up permissions from original file
	t="${r}.$$"
	execute "cp -p ${f} ${t}"
	if [ $? -ne 0 ]; then
		exit 1
	fi
	execute "chmod +w ${t}"

	# Transform
	execute "exiftran ${option} -o ${t} ${f}"
	if [ $? -ne 0 ]; then
		rm ${t}
		exit 1
	fi

	# Do this last in case exiftran fails
	if [ ${override} -eq 0 ]; then
		# Don't keep original when overriding
		execute "mv ${f} ${o}"
	else
		# Pick up timestamps from original
		o="${o%r\.jpg-}.jpg-"
	fi

	# Rename temporary then copy access and modified timestamps
	execute "mv ${t} ${r}"
	execute "${touch} -r ${o} ${r}"
	if [ ${override} -eq 0 ]; then
		echo "${prog}: ${f} -> ${r}"
	else
		echo "${prog}: ${r} (rotated)"
	fi
done

exit ${status}
